//
//							noiseblanker.v - Noise Blanker
//
//						(C) Copyright 2009 John B. Stephensen
//
// This Verilog source file and all its derivatives are licensed only for
// personal non-profit educational use in the Amateur Radio Service and
// the license is not transferrable. The information is provided as-is for
// experimental purposes and the author does not warranty its freedom
// from defects or its suitability for any specific application.
//
// This module processes pairs of samples and zeroes both samples if one sample
// is larger than a preset threshold. Samples arrive on DIXY and leave via DOXY.
// Samples are clocked by DCLK with the X sample present when SCLK = 1 and the
// Y sample present when SCLK = 0; Blanking is accomplished in 2 SCLK cycles.
// Note that four 18-bit registers must be instantiated to prevent optimization
// from altering timing.
// 
// 55 slices are used and the maximum clock rate is 153 MHz for SCLK and 758 MHz
// for DCLK.
//
// Normal Warnings:
//		Inputs <cin<15>> and <cin<6:0>> are never used.
//
module noiseblanker(
    input [17:0] dixy,	// data input (X when SCLK = 1, Y when SCLK = 0)
    input iv,				// input valid
    output [17:0] doxy,	// data output (X when SCLK = 1, Y when SCLK = 0)
    output ov,				// output valid
    input sclk,			// sample clock
    input dclk,			// double-rate clock
    input mrst,			// master reset
    input [15:0] cin,	// configuration information
    input cwe				// write enable
    );
// internal signals
reg [7:0] limit;			// threshold
wire [7:0] mag;			// magnitude of upper 9 bits of input
reg [2:1] v;				// delayed sample valid flags
wire b0;						// greater than limit
reg b1,b2;					// delayed version
wire [17:0] s0,s1,s2;	// sample registers
// configuration register
always @ (posedge sclk)
begin
	if (mrst) limit <= 8'hFF;
	else if (cwe) limit <= cin[14:7];
end
// get absolute value of signal
tc8s absv (
	.D(dixy[16:9]),	// signal
	.CMP(dixy[17]),	// sign bit
	.Q(mag),				// magnitude of signal
	.CLK(dclk),
	.CE(iv),				// sample when input valid
	.SCLR(mrst)
	);
// detect noise pulses by comparing with threshold
always @ (posedge sclk)
begin
	v <= {v[1],iv};				// X or Y sample valid delay
end
assign b0 = (mag > limit);
always @ (posedge sclk)
begin
	if (iv) b1 <= b0;			// save magnitude
end
always @ (negedge sclk)
begin
	if (v[1]) b2 <= b0 | b1;	// blank if X or Y above threshold
end
reg18 r0 (
	.d(dixy),
	.q(s0),
	.ce(1'b1),
	.clk(dclk),
	.rst(1'b0)
	);
reg18 r1 (
	.d(s0),
	.q(s1),
	.ce(1'b1),
	.clk(dclk),
	.rst(1'b0)
	);
reg18 r2 (
	.d(s1),
	.q(s2),
	.ce(1'b1),
	.clk(dclk),
	.rst(1'b0)
	);
reg18 r3 (
	.d(s2),
	.q(doxy),
	.ce(1'b1),
	.clk(dclk),
	.rst(b2)
	);
// connect outputs
assign ov = v[2];
endmodule
