/*
 * twlog:  A basic ham loging program using Motif
 * Copyright (C) 1997 Ted Williams WA0EIR (ted@bluestone.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge. MA 02139,
 * USA. See the COPYING file in this directory.
 *
 * Versions: 1.1 -  Oct 1997 
 */

/*
 * DIALOGS FOR TWLOG AND THEIR CALLBACKS
 * 
 */

#include "twlog.h"

#define PBWIDTH  80
#define MARGIN   20

struct tag2
          {
           Widget searchTF;
           Widget editTxt;
           Boolean reset;
          };

/*
 * popupEdit Function
 * This dialog is the log editor.  It is created the first time it is
 * called, and just popped up and down after that by managing its form.
 * It is application-modal to prevent the main interface from writing
 * to the logfile while the editor has it open.
 */
void popupEdit (char *filename)
{
   static Widget editDiag, searchTF, sep1, okPB, savePB;
   static Widget searchPB, form2, editTxt, searchLabel;
   static struct tag2 textwids;

   if (editDiag == NULL)
      {
      editDiag = XtVaCreateWidget ("editDiag", xmDialogShellWidgetClass,  logSH,
                    XmNtitle, "TWLOG EDITOR",
                    XmNmwmDecorations,
	               MWM_DECOR_RESIZEH | MWM_DECOR_TITLE | MWM_DECOR_BORDER,
	            XmNminWidth, 300,
	            XmNminHeight, 300, 
                    NULL);

      form2 = XtVaCreateWidget ("form2", xmFormWidgetClass, editDiag,
                 XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
                 XmNautoUnmanage, False,
                 NULL);

      okPB = XtVaCreateManagedWidget ("OK", xmPushButtonWidgetClass, form2,
                XmNwidth, PBWIDTH,
                XmNbottomAttachment, XmATTACH_FORM,
                XmNbottomOffset, MARGIN,
                XmNleftAttachment, XmATTACH_FORM,
                XmNleftOffset, MARGIN,
                NULL);

      savePB = XtVaCreateManagedWidget ("Save", xmPushButtonWidgetClass, form2,
                  XmNwidth, PBWIDTH,
                  XmNbottomAttachment, XmATTACH_FORM,
                  XmNbottomOffset, MARGIN,
                  XmNleftAttachment, XmATTACH_POSITION,
                  XmNleftPosition, 50,
                  XmNleftOffset, -PBWIDTH / 2,
                  NULL);

      searchPB = XtVaCreateManagedWidget ("Search", xmPushButtonWidgetClass, form2,
                    XmNwidth, PBWIDTH,
                    XmNbottomAttachment, XmATTACH_FORM,
                    XmNrightAttachment, XmATTACH_FORM,
                    XmNbottomOffset, MARGIN,
                    XmNrightOffset, MARGIN,
                    NULL);

      sep1 = XtVaCreateManagedWidget ("sep1", xmSeparatorWidgetClass, form2,
                XmNtopAttachment, XmATTACH_NONE,
                XmNbottomAttachment, XmATTACH_FORM,
                XmNbottomOffset, 70,
                XmNleftAttachment, XmATTACH_FORM,
                XmNrightAttachment, XmATTACH_FORM,
                NULL);

      searchTF = XtVaCreateManagedWidget ("searchTF", xmTextFieldWidgetClass, form2,
                    XmNtopAttachment, XmATTACH_NONE,
                    XmNbottomAttachment, XmATTACH_WIDGET,
                    XmNbottomWidget, sep1,
                    XmNbottomOffset, MARGIN,
                    XmNleftAttachment, XmATTACH_FORM,
                    XmNrightAttachment, XmATTACH_FORM,
                    XmNleftOffset, MARGIN,
                    XmNrightOffset, MARGIN,
                    NULL);

      searchLabel = XtVaCreateManagedWidget ("Search Text:", xmLabelWidgetClass, form2,
                     XmNbottomAttachment, XmATTACH_WIDGET,
                     XmNbottomWidget, searchTF,
                     XmNleftAttachment, XmATTACH_FORM,
                     XmNleftOffset, MARGIN,
                     NULL);

      editTxt = XmCreateScrolledText (form2, "editTxt", NULL, 0);
      XtVaSetValues (editTxt,
         XmNcolumns, 80,
         XmNrows, 12,
         XmNeditMode, XmMULTI_LINE_EDIT,
         NULL);

      XtVaSetValues (XtParent(editTxt),
         XmNtopAttachment, XmATTACH_FORM,
         XmNtopOffset, MARGIN,
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomWidget, searchLabel,
         XmNbottomOffset, MARGIN,
         XmNleftAttachment, XmATTACH_FORM,
         XmNleftOffset, MARGIN,
         XmNrightAttachment, XmATTACH_FORM,
         XmNrightOffset, MARGIN,
         NULL);
	 
      /*
       * Set up the struct that is passed as client data
       */
      textwids.searchTF = searchTF;
      textwids.editTxt = editTxt;
      textwids.reset = True;

      XtAddCallback (okPB, XmNactivateCallback, okCB, (XtPointer) form2);
      XtAddCallback (savePB, XmNactivateCallback, saveCB, (XtPointer) textwids.editTxt);
      XtAddCallback (searchPB, XmNactivateCallback, searchCB, (XtPointer) &textwids);
      XtAddCallback (searchTF, XmNactivateCallback, searchCB, (XtPointer) &textwids);
      XtAddCallback (searchTF, XmNmodifyVerifyCallback, searchCB, (XtPointer) &textwids);
      XtManageChild (editTxt);
      }
   XtManageChild (form2);
   
   /*
    *Read the file and put it in the widget
    */
   getdata (filename, editTxt);
}


/*
 * popupHelp Function
 * Created the first time it is called and then managed as needed.  
 * Set for modeless so main interface is still active.
 */
void popupHelp (char *filename)
{
static Widget helpDiag, helpTxt;
XmString xs;

   if (helpDiag == NULL)
      {
      helpDiag = XmCreateInformationDialog(logSH, "helpDiag", NULL, 0);
      XtVaSetValues (XtParent(helpDiag),
         XmNtitle, "TWLOG HELP FILE",
         NULL);
      XtUnmanageChild (XmMessageBoxGetChild (helpDiag, XmDIALOG_CANCEL_BUTTON));
      XtUnmanageChild (XmMessageBoxGetChild (helpDiag, XmDIALOG_HELP_BUTTON));

      xs = XmStringCreateLocalized ("TWLOG HELP FILE");
      XtVaSetValues (helpDiag,
         XmNdialogStyle, XmDIALOG_MODELESS,
         XmNmessageString, xs,
         NULL);
      XmStringFree(xs);

      helpTxt = XmCreateScrolledText (helpDiag, "helpTxt", NULL, 0);
      XtVaSetValues (helpTxt,
         XmNcolumns, 80,
         XmNmargin, 10, 
         XmNrows, 10,
         XmNeditMode, XmMULTI_LINE_EDIT,
         XmNeditable, False,
         XmNresizeWidth, True,
         XmNautoShowCursorPosition, False,
         XmNcursorPositionVisible, False,
         NULL);
      XtManageChild (helpTxt);
      }
   getdata (filename, helpTxt);
   XtManageChild (helpDiag);
}


/*
 * getdata Function
 * Passed a filename and a textfield widget, this will open the file,
 * read it, and put the data into the textfield widget.  Used by both
 * edit and help dialog.
 */
void getdata (char *filepath, Widget txtwid)
{
   struct stat file_stat;
   char *text_ptr = NULL;
   FILE *fp;

   if (stat (filepath, &file_stat) != 0)
      {
      perror ("getdata - stat failed");
      fprintf (stderr, "Filename: %s\n", filepath);
      exit (1);
      }

   if ((text_ptr = (char *) XtMalloc ((int)file_stat.st_size + 1)) == NULL)
      {
      perror ("getdata - malloc failed");
      exit (1);
      }

   if ((fp = fopen (filepath, "r")) == NULL)
      {
      perror ("getdata - open failed");
      fprintf (stderr, "Filename: %s\n", filepath);
      exit (1);
      }

   if (fread (text_ptr, 1, file_stat.st_size, fp) != (int)(file_stat.st_size))
      {
      perror ("getdata - fread failed");
      fprintf (stderr, "Filename: %s\n", filepath);
      exit (1);
      }

   text_ptr[file_stat.st_size] = '\0';
   fclose (fp);

   XtVaSetValues (txtwid,
      XmNvalue, text_ptr,
      NULL);
   XtFree (text_ptr);
}


/*
 * okCB Callback Function 
 * Used to unmanage the edit dialog.  The about and help dialog don't use this.
 */
void okCB (Widget w,  XtPointer cdata, XtPointer cbs)
{
   XtUnmanageChild ((Widget) cdata); 
}


/*
 * saveCB Callback Function 
 * writes contents of edit diag to logfile
 */
void saveCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   char *txt_ptr = NULL;
   FILE *fp;
   Widget wid = (Widget) cdata;

   XtVaGetValues (wid,
      XmNvalue, &txt_ptr,
      NULL);
   if ((fp = fopen (logpath, "w")) == NULL)
      {
      perror("saveCB - open failed");
      exit (1);
      }
   fwrite (txt_ptr, strlen (txt_ptr), 1, fp);
   fclose (fp);
}


/*
 * searchCB Callback Function
 * Activate and modify/verify callback for the Search button and textfield.
 * For an activate, get the string, go look for it and if found, highlight
 * the text and set the top position so the text will be in the window.
 * If not found, make reset = True and clear the search text widget.
 * If it was a modify/verify from the searchTF then the search pattern has
 * changed. Set the reset flag to True so we start the search over.
 */
void searchCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   struct tag2 *wids = (struct tag2 *) cdata;
   static XmTextPosition start_pos, found_pos, end_pos;
   char *find_str;
   XmAnyCallbackStruct *cbsTF = (XmAnyCallbackStruct *) cbs;

   XmTextSetHighlight (wids->editTxt, found_pos, end_pos, XmHIGHLIGHT_NORMAL);

   if (wids->reset == True)
      start_pos = found_pos = 0;

   switch (cbsTF->reason)
      {
      case XmCR_ACTIVATE:                 
         XtVaGetValues (wids->searchTF,
            XmNvalue, &find_str,
            NULL);
         if (XmTextFindString (wids->editTxt, start_pos, find_str, XmTEXT_FORWARD, &found_pos))
            {
            XmTextSetInsertionPosition (wids->editTxt, found_pos);
            end_pos = found_pos + strlen (find_str);
            XmTextSetHighlight (wids->editTxt, found_pos, end_pos, XmHIGHLIGHT_SELECTED);
            start_pos = end_pos + 1; 
            wids->reset = False;
            XmTextSetTopPosition (wids->editTxt, found_pos);
            }
         else                             
            {
            wids->reset = True;
            XtVaSetValues (wids->searchTF,
               XmNvalue, "",
               NULL);
            }
         break;

      case XmCR_MODIFYING_TEXT_VALUE:
         /*on mod/ver, make reset true so we start over*/
         wids->reset = True;
         break;
      }
}


/*
 * aboutDiag Function
 * Just an info dialog
 */
void aboutDiag ()
{
   static Widget diag = NULL;
   XmString  msg_xs;
   char msg_str[] = "twlog \251 1997 was written by\nTed Williams - WA\330EIR";

   msg_xs = XmStringCreateLtoR (msg_str, XmFONTLIST_DEFAULT_TAG);
   if (diag == NULL)
      {
      diag = XmCreateInformationDialog (logSH, "aboutDiag",  (ArgList) NULL, 0);
      XtVaSetValues (XtParent(diag),
         XmNtitle, "ABOUT TW LOG",
         NULL);

      XtVaSetValues (diag,
         XmNmessageAlignment, XmALIGNMENT_CENTER,
         XmNmessageString, msg_xs,
         NULL);
      XmStringFree(msg_xs);

      XtUnmanageChild (XmMessageBoxGetChild (diag, XmDIALOG_CANCEL_BUTTON));
      XtUnmanageChild (XmMessageBoxGetChild (diag, XmDIALOG_HELP_BUTTON));
      }
   XtManageChild (diag);
}
