#include "global.h"
#ifdef STATS_FWD
#include "stats.h"
#include "stats_f.h"
#ifdef MSDOS
#include "ctype.h"
#endif

#if !defined(_lint)
static char rcsid[] OPTIONAL = "$Id: stats_f.c,v 1.12 1997/07/31 00:44:20 root Exp root $";
#endif

static int STATdaily_fwd (int argc,char *argv[],void *p);
static int STATweekly_fwd (int argc,char *argv[],void *p);
static int STATmonthly_fwd (int argc,char *argv[],void *p);
static int STATyearly_fwd (int argc,char *argv[],void *p);
static int STATgeneral_fwd (int argc,char *argv[],void *p);
static int STATlist_fwd (int argc,char *argv[],void *p);

static struct fwd_stats fwdstats;


static struct fwd_data *sanity_check (int argc,char *argv[],void *p);
static struct fwd_data *find_entry (char *name);
static struct fwd_data *create_entry (char *name, int add);


static const char fwddailyhdr[] = "%8.8s%-35.35sOUTGOING\n\n";
static const char fwdweeklyhdr[] = "%12.12s%-40.40sOUTGOING\n\n";
static const char incomingstr[] = "INCOMING";
static const char fwdwithstr[] = "Forwarding with %s";
static const char fwdmonthlyhdr[] = "%12.12s%-42.42sOUTGOING\n\n";
static const char fwdyearlyhdr[] = "%2.2s%-23.23sOUTGOING\n\n";


static struct cmds STATSfwdcmds[] = {
	{ "clear",	doSTATclear_fwd,0,	0,	NULLCHAR },
	{ "daily",	STATdaily_fwd,	0,	0,	NULLCHAR },
	{ "general",	STATgeneral_fwd,0, 	0,	NULLCHAR },
	{ "list",	STATlist_fwd,	0, 	0,	NULLCHAR },
	{ "monthly",	STATmonthly_fwd,0, 	0,	NULLCHAR },
	{ "weekly",	STATweekly_fwd,	0, 	0,	NULLCHAR },
	{ "yearly",	STATyearly_fwd,	0, 	0,	NULLCHAR },
	{ NULLCHAR,	NULL,		0,	0,	NULLCHAR }
};



int
doSTATforward(argc,argv,p)
int argc;
char *argv[];
void *p;
{
char *temp;

	if (argc == 3)	{
		temp = argv[1];
		argv[1] = argv[2];
		argv[2] = temp;
	} else if (argc == 2 && tolower(argv[1][0]) != 'l' )
		tprintf ("Syntax: stats forwarding <bbsname> <subcommand>\n\n");
	return subcmd(STATSfwdcmds,argc,argv,p);
}


static struct fwd_data *
find_entry (name)
char *name;
{
struct fwd_data *fd;

	fd = fwdstats.head;
	while (fd)	{
		if (!stricmp (name, fd->name))
			return (fd);
		fd = fd->next;
	}
	return ((struct fwd_data *)-1);
}


static struct fwd_data *
create_entry (name, add)
char *name;
int add;
{
struct fwd_data *fd;

	fd = find_entry (name);
	if (fd == (struct fwd_data *)-1)	{
		fd = callocw (1, sizeof (struct fwd_data)); /* no error checking, yet */
		if (fd != (struct fwd_data *)0)	{
			fd->name = strdup (name);
			(void) strupr (fd->name);
			fd->next = fwdstats.head;
			fwdstats.head = fd;
			if (add)
				fwdstats.count++;
		}
	}
	return (fd);
}


static struct fwd_data *
sanity_check(argc,argv,p)
int argc;
char *argv[];
void *p OPTIONAL;
{
struct fwd_data *fd;

	if (argc < 2)	{
		tprintf ("This command requires both a 'bbsname' and a subcommand\n");
		return ((struct fwd_data *)-1);
	}
	
	fd = find_entry(argv[1]);
	if (fd == (struct fwd_data *)-1)
		tprintf ("Statistics data for that BBS not found\n");
	return (fd);
}


int
doSTATclear_fwd(argc,argv,p)
int argc OPTIONAL;
char *argv[] OPTIONAL;
void *p OPTIONAL;
{
struct fwd_data *fd, *temp;

	if(Curproc->input != Command->input)
		tputs (STAT_cannotclear);
	else	{
		fd = fwdstats.head;
		while (fd)	{
			free (fd->name);
			temp = fd;
			fd = fd->next;
			free (temp);
		}

		memset (&fwdstats, 0, sizeof (struct fwd_stats));
		fwdstats.days = 1;
		fwdstats.start = time((time_t *)0);
		savestats_fwd();
		log (-1, "Clearing forward stats");
	}
	return 0;
}


static int
STATlist_fwd(argc,argv,p)
int argc OPTIONAL;
char *argv[] OPTIONAL;
void *p OPTIONAL;
{
struct fwd_data *fd;
int theindex = 0;

	tputs ("BBSs with defined statistics:\n\n");
	fd = fwdstats.head;
	while (fd)	{
		tprintf ("%-19.19s", fd->name);
		if (!(++theindex % 4))
			tputc ('\n');
		fd = fd->next;
	}
	tputc ('\n');
	if (theindex % 4)
		tputc ('\n');
	return 0;
}


static int
STATgeneral_fwd(argc,argv,p)
int argc;
char *argv[];
void *p;
{
char buf1[45], buf2[26], buf3[36];
int temp, l;
int highest[2], second[2];
struct fwd_data *fd;

	if ((fd = sanity_check (argc, argv, p)) == (struct fwd_data *)-1)
		return 0;

	sprintf (buf1, "FORWARDING with %s", fd->name);
	tprintf ("\n%26.26s: %-26.26s%-24.24s\n", buf1, incomingstr, "OUTGOING");
	sprintf (buf1, "%ld", fd->messages[0]);
	sprintf (buf2, "%ld", fd->messages[1]);
	sprintf (buf3, "Messages since %6.6s", &(ctime(&fwdstats.start))[4]);
	tprintf ("  %24.24s: %-26.26s%-24.24s\n", buf3, buf1, buf2);
	sprintf (buf1, "%ld", fd->day[0]);
	sprintf (buf2, "%ld", fd->day[1]);
	tprintf ("  %24.24s: %-26.26s%-24.24s\n", "Messages since midnight",
		buf1, buf2);
	sprintf (buf1, "%ld", fd->hour[0]);
	sprintf (buf2, "%ld", fd->hour[1]);
	tprintf ("  %24.24s: %-26.26s%-24.24s\n", "Messages this hour",
		buf1, buf2);
	sprintf (buf1, "%d", (fwdstats.days > 1) ? (int)((fd->messages[0] - fd->day[0] + (fwdstats.days - 2)) / (fwdstats.days - 1)) : (int)fd->messages[0]);
	sprintf (buf2, "%d", (fwdstats.days > 1) ? (int)((fd->messages[1] - fd->day[1] + (fwdstats.days - 2)) / (fwdstats.days - 1)) : (int)fd->messages[1]);
	tprintf ("  %24.24s: %-26.26s%-24.24s\n", "Average messages per day",
		buf1, buf2);
	highest[0] = highest[1] = -1;
	second[0] = second[1] = -1;
	for (temp = 0; temp < 24; temp++)
		for (l = 0; l < 2; l++)	{
			if (highest[l] == -1 || fd->hourly[temp][l] > fd->hourly[highest[l]][l])	{
				second[l] = highest[l];
				highest[l] = temp;
			} else if (second[l] == -1 || fd->hourly[temp][l] > fd->hourly[second[l]][l])
				second[l] = temp;
		}
	sprintf (buf1, "At %02d and %02d o'clock", highest[0], second[0]);
	sprintf (buf2, "At %02d and %02d o'clock", highest[1], second[1]);
	tprintf ("%26.26s: %-26.26s%-24.24s\n\n", "Rush hours", buf1, buf2);
	return 0;
}



static int
STATdaily_fwd(argc,argv,p)
int argc;
char *argv[];
void *p;
{
char buf[45];
struct fwd_data *fd;

	if ((fd = sanity_check (argc, argv, p)) == (struct fwd_data *)-1)
		return 0;
	sprintf (buf, fwdwithstr, fd->name);
	tprintf (dailyhdr, buf);
	tprintf (fwddailyhdr, STAT_emptystr,incomingstr);
	doGraph (24, fd->hourly);
	tputs (STAT_dailytrailer);
	return 0;
}


static int
STATweekly_fwd(argc,argv,p)
int argc;
char *argv[];
void *p;
{
char buf[45];
struct fwd_data *fd;

	if ((fd = sanity_check (argc, argv, p)) == (struct fwd_data *)-1)
		return 0;
	sprintf (buf, fwdwithstr, fd->name);
	tprintf (weeklyhdr, buf);
	tprintf (fwdweeklyhdr, STAT_emptystr,incomingstr);
	doGraph (7, fd->daily);
	tputs (STAT_weeklytrailer);
	return 0;
}


static int
STATmonthly_fwd(argc,argv,p)
int argc;
char *argv[];
void *p;
{
char buf[45];
struct fwd_data *fd;

	if ((fd = sanity_check (argc, argv, p)) == (struct fwd_data *)-1)
		return 0;
	sprintf (buf, fwdwithstr, fd->name);
	tprintf (monthlyhdr, buf);
	tprintf (fwdmonthlyhdr, STAT_emptystr,incomingstr);
	doGraph (31, fd->monthly);
	tputs (STAT_monthlytrailer);
	return 0;
}


static int
STATyearly_fwd(argc,argv,p)
int argc;
char *argv[];
void *p;
{
char buf[45];
struct fwd_data *fd;

	if ((fd = sanity_check (argc, argv, p)) == (struct fwd_data *)-1)
		return 0;
	sprintf (buf, fwdwithstr, fd->name);
	tprintf (yearlyhdr, buf);
	tprintf (fwdyearlyhdr, STAT_emptystr,incomingstr);
	doGraph (12, fd->yearly);
	tputs (STAT_yearlytrailer);
	return 0;
}


void
STATS_addfwd (int which, int amount, char *name)
{
struct fwd_data *fd;

	if (which < 0 || which > 1)
		return;
	fd = create_entry (name, 1);
	fd->messages[which] += amount;
	fd->hour[which] += amount;
	fd->day[which] += amount;
	fd->month[which] += amount;
}


void
updatestats_fwd (void)
{
	/* nothing, at this time - addfwdstats() does all the work */
}


void
loadstats_fwd(void)
{
FILE *fp;
char buffer[256];
struct fwd_data *fd;
int count;
int len;

	sprintf (buffer, "%s/forward.dat", STATSDir);
	fp = fopen (buffer, "r");
	if (fp != NULLFILE)	{
		(void) fread (&fwdstats, sizeof (struct fwd_stats) - sizeof(struct fwd_data *), 1, fp);
		for (count = 0; count < fwdstats.count; count++)	{
			len = fgetc (fp);
			(void) fread (buffer, (unsigned) len, 1, fp);
			buffer[len] = 0;
			fd = create_entry (buffer, 0);
			(void) fread (fd->messages, sizeof (struct fwd_data) - (sizeof(char *) + sizeof(struct fwd_data *)), 1, fp);	/*lint !e419 */
		}
		(void) fclose (fp);
	}
}


void
savestats_fwd(void)
{
FILE *fp;
char buffer[256];
struct fwd_data *fd;
int count;

	sprintf (buffer, "%s/forward.dat", STATSDir);
	fp = fopen (buffer, "w");
	if (fp != NULLFILE)	{
		(void) fwrite (&fwdstats, sizeof (struct fwd_stats) - sizeof(struct fwd_data *), 1, fp);
		fd = fwdstats.head;
		for (count = 0; count < fwdstats.count; count++, fd = fd->next)	{
			fprintf (fp, "%c%s", strlen (fd->name), fd->name);
			(void) fwrite (fd->messages, sizeof (struct fwd_data) - (sizeof(char *) + sizeof(struct fwd_data *)), 1, fp);	/*lint !e420 */
		}
		(void) fclose (fp);
	} else
		log (-1, "Can't open stats file '%s/forward.dat'", STATSDir);
}


void
newhour_fwd (int hour)
{
struct fwd_data *fd;

	fd = fwdstats.head;
	while (fd)	{
		fd->hourly[hour][0] = fd->hour[0];
		fd->hourly[hour][1] = fd->hour[1];
		fd->hour[0] = fd->hour[1] = 0;
		fd = fd->next;
	}
}


void
newday_fwd (int day)
{
struct fwd_data *fd;

	fd = fwdstats.head;
	while (fd)	{
		fd->daily[day][0] = fd->day[0];
		fd->daily[day][1] = fd->day[1];
		fd = fd->next;
	}
}


void
endmonthclear_fwd (int day, int month)
{
int k;
struct fwd_data *fd;
FILE *fp;

	fd = fwdstats.head;
	fp = tmpfile();
	if (fp)
		fprintf (fp, "Monthly Forwarding Stats for the month of %s: %s\n\n", Months[month], Hostname);
	while (fd)	{
		/* clear out non-existent days of last month */
		for (k = day; k < 31; k++)
			fd->monthly[k][0] = fd->monthly[k][1] = 0L;
		fd->yearly[month][0] = fd->month[0];
		fd->yearly[month][1] = fd->month[1];
		if (fp)
			fprintf (fp, "[%s] Incoming: %ld\n[%s] Outgoing: %ld\n", fd->name, fd->month[0], fd->name, fd->month[1]);
		fd->month[0] = fd->month[1] = 0;
		fd = fd->next;
	}
	if (fp)	{
		rewind (fp);
		(void) rdaemon (fp, NULLCHAR, NULLCHAR, "sysop", "Monthly Forwarding Stats", 'P', 0);
		(void) fclose (fp);
	}
}


void
endday_fwd (int day)
{
struct fwd_data *fd;

	fd = fwdstats.head;
	while (fd)	{
		fd->monthly[day][0] = fd->day[0];
		fd->monthly[day][1] = fd->day[1];
		fd->day[0] = fd->day[1] = 0;
		fd = fd->next;
	}
	fwdstats.days++;
}


void
eachcycle_fwd (time_t now)
{
	fwdstats.last = now;
}


void
init_fwd (time_t now)
{
	fwdstats.days = 1;
	fwdstats.start = now;
}


#endif /* STATS_FWD */
