// FirLp.cpp: implementation of the CFirLp class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "FirLp.h"
#include "math.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CFirLp::CFirLp(int nSize, double dCutOff)
{
	Common(nSize,dCutOff,0.54,0.46, 0.0);	// Default to Hamming window
}
//////////////////////////////////////////////////////////////////////
CFirLp::~CFirLp()
{
	delete[] dpHBase;
	delete[] dpBase; 
}
//////////////////////////////////////////////////////////////////////
CFirLp::CFirLp(int nSize, double dCutOff, int w)
{
	if (w == HAMMING)
		Common(nSize,dCutOff,0.54,0.46, 0.0);// Default to Hamming window
	else  if ((w == HANNING) || (w == RAISEDCOSINE))
			Common(nSize,dCutOff,0.5,0.5, 0.0);	// Hanning window (raised cosine)
	else  if ( w == BLACKMAN)
			Common(nSize,dCutOff,0.42,0.5,0.08);// Blackman window
	else
		AfxMessageBox("FirLp: Type of windowing request is not valid");
}
//////////////////////////////////////////////////////////////////////
CFirLp::CFirLp(int nSize, double dCutOff, double dPrm1, double dPrm2, double dPrm3)
{
	Common(nSize,dCutOff, dPrm1, dPrm2, dPrm3);
}

////////////////////////////////////////////////////////////
// Setup arrays and compute coefficients
////////////////////////////////////////////////////////////
CFirLp::Common(int nSize, double dCutOff, double dP1, double dP2, double dP3)
{
	int i;
	double pi = 3.1415926535897932384626433;

	nHalf = nSize/2;		// Rounded-down, half size
	nBit0 = (nSize & 0x01);
	nHalf1 = nBit0 + nHalf;	// Handle nSize odd case.

// Allocate an array of doubles
	dpBase = new double[nSize];// Low end of data array

// Allocate an array for filter coefficients 
// (symmetrical, therefore only half size required)
	dpHBase = new double[nHalf1];	// Low end of coefficient array


// Compute coefficients for filter
// and scale with window function
	double dOmegaC = dCutOff * pi * 2.0;
	double dI,dS;
	double dOffSet = 0.5 - (float(nSize)/2);

	for (i = 0; i < nHalf1; i++)
	{
		dI = (i + dOffSet);
		dS = dI * dOmegaC;
		if (dS < 1E-12) 
		{
	// Basic rectangular filter
			*(dpHBase + i) = sin(dS)/dS;
		}
		else
			*(dpHBase + i) = 1.0;
	// Apply windowing
		*(dpHBase + i) *= dP1 - dP2 * cos( (2 * pi * i)/(nSize-1) ) +
			dP3 * cos( (4 * pi * i)/(nSize-1) );
	}

// Scale so that gain (total) of filter equals 1.00
	double dX = 0;

	for (i = 0; i < nHalf; i++)// Sum original array
	{
		dX += *(dpHBase + i);
	}
	dX *= 2;	// Symmetrical, so count both sides.
	if ( nBit0 != 0 )
	{	// Handle odd case
		dX += *(dpHBase + i);
	}

	for (i = 0; i < nHalf1; i++)// Scale array
		*(dpHBase + i) *= 1.0/dX;

// Zero stored values	
	double *d = dpBase;
	for (i = 0; i < nSize; i++)
		*d++ = 0;

	dpHiEnd = d;	// Save high end of array

	nY = 0;			// Zero data array index

	nSze = nSize;	// Save size for later 'Fir' filtering

}
////////////////////////////////////////////////////////////
// Remove old data pt and insert new 'dData' to storage,
// then compute and return new filtered output.
////////////////////////////////////////////////////////////
double CFirLp::Fir(double dData)
{
	int j,jj;
	double dSum = 0;
	
	j = nY;				// j  works towards increasing indices
	jj = nY - 1;		// jj works with decreasing indices
	*(dpBase + nY++) = dData;	// Circular buffer incoming data
	if (nY >= nSze) nY = 0;	// Check for wrap around
	if (jj < 0) jj = nSze-1;	// Check for wrap around

	/* 
	At this point nY points to next-available-storage location
	j  points to most recent stored data point
	jj points to oldest stored data point.
	*/
	
	/* 
	Multiply and accumulate each data pt in the incoming circular buffer 
	with each coefficient.  Since coefficients are symmetrical, reuse them
	by working from outside to inside ends of circular buffered data.  In so
	doing, the precision issue is helped as the results of the multiplies to the
	smallest coefficients are added first, thus reducing the loss of precision 
	by adding small (floating) numbers to big ones.
	*/
	
	for ( int i = 0; i < nHalf; i++)
	{
		dSum = dSum + (*(dpHBase + i) * *(dpBase + j++)  );
		if (j >= nSze) j = 0;		
		dSum = dSum + (*(dpHBase + i) * *(dpBase + jj--) );
		if (jj < 0) jj = nSze-1;
	}
	if ( nBit0 != 0 )
	{	// Here, odd number of taps
		dSum = dSum + (*(dpHBase + i) * *(dpBase + j++)  );
		if (j >= nSze) j = 0;		
	}

	return dSum;

}
////////////////////////////////////////////////////////////
// Fill array pointed to by 'pd' with coefficients
////////////////////////////////////////////////////////////
int CFirLp::GetCoefficients(double *pd, int nSz)
{
	int j = nHalf1;
// Use smallest size jic we get passed a size that
// too big (such as forgetting that the coefficient
// array is 1/2 or 1/2 + 1 the size of the number
// taps in the filter.
	if (nSz < j) 
		j = nSz;
	for (int i = 0; i < j; i++)
		*(pd + i) = *(dpHBase + i);
	return j;
}
