unit Dds_chip;      {18.8.1995, Bernd Hoen}
                    {DivMul: TDivMul}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls;

const
  DDS_Clock  = 0;
  DDS_Data   = 1;
  DDS_Load   = 2;
  DDS_TC1    = 3;
  LastBit=32;     {Last bit of phase accumulator}

type
    tPortNumber = 1..2;  {1=LPT1, 2=LPT2}
    tDivMul     = (Divide, Multiply); {affects property Divider}


type
  TDdsChip = class(TComponent)
  private
    {-----Fields-----}
    io_adresse : word;  { Lpt1=$0378, Lpt2=$037A }
    Buffer     : byte;
    SelectedPortNumber : integer;

    FFtw1, FFtw0   : Longint;
    FQuartz        : Extended;
    FDivider       : Longint;
    FDivMul        : tDivMul;
    FBits          : Integer;
    FClock,
    FHardwareMin,
    FHardwareMax,
    FUserMin, FUserMax,
    FFrequency1, FFrequency0,
    ZweiHochBitzahl            : Extended;
    {-----pointer to user code-----------}
    FOnParamError:       TNotifyEvent;
    FOnUserMinMaxChange: TNotifyEvent;
    {-----pointer to user components-----}
    FQuartzEdit,
    FDividerEdit,
    FBitsEdit     : TEdit;
    FClockLabel   : TLabel;
    FHardwareMinLabel, FHardwareMaxLabel,
    FUserMinLabel,     FUserMaxLabel      : TLabel;
    FFrequency1Edit,   FFrequency0Edit    : TEdit;
    {-----methods------------------------}
    Procedure SetQuartz(F:Extended);
    Procedure SetDivMul(value: tDivMul);
    Procedure SetDivider(D:Longint);
    Procedure SetBits(B:Integer);
    Procedure SetClockDependentVariables;
    Procedure SetFrequency1(F:Extended);
    Procedure SetFrequency0(F:Extended);
    Procedure SendPhase(TC:integer);
    {paraport}
    function Init_Port(PortNumber:tPortNumber):boolean;     {True if OK}
    function Test_Port(PortNumber:tPortNumber):boolean;     {True if OK}
  protected
    Procedure ParamError; dynamic;       {new method}
    Procedure UserMinMaxChange; dynamic; {new method}
  public
    constructor Create(AOwner: TComponent); override;
    Function HardwareMin:Extended;
    Function HardwareMax:Extended;
    Function UserMin:Extended;
    Function UserMax:Extended;
    Procedure SetUserMinMax(Fmin,Fmax:Extended);
    Function Clock: Extended;
    Function Limited(Value, min, max: Extended):Extended;
    Function FreqStr(Frequency:Extended):String;
    Function GetFTW(TC:integer):Longint;
    {parport methods}
    Function  Port1Available:boolean;             { 1. Abfragen. (Muss nicht sein)}
    Function  Port2Available:boolean;

    Function  SelectPort(Number:tPortNumber):boolean; { 2. Whlen  (true=OK)}
    Function  SelectedPort:integer;

    procedure Port_Out_Byte(b:byte);                      { 3. und los gehts...}
    procedure Port_Set_Bit(Nummer:byte);
    procedure Port_Reset_Bit(Nummer:byte);
    function  Port_Get_Bit(Nummer:byte):boolean;
    procedure SendBits(TC1:integer; Phase:longint; AnzahlBits:integer);

  published
    {-----Fields-----}
    Property Quartz     : Extended read FQuartz       write SetQuartz;
    Property DivMul     : tDivMul  read FDivMul       write SetDivMul;
    Property Divider    : Longint  read FDivider      write SetDivider;
    Property Bits       : Integer  read FBits         write SetBits;
    Property Frequency1 : Extended read FFrequency1   write SetFrequency1;
    Property Frequency0 : Extended read FFrequency0   write SetFrequency0;
    {-----Component slots-----}
    Property QuartzEdit    : TEdit  read FQuartzEdit     write FQuartzEdit;
    Property DividerEdit   : TEdit  read FDividerEdit    write FDividerEdit;
    Property ClockLabel    : TLabel read FClockLabel     write FClockLabel;
    Property BitsEdit      : TEdit  read FBitsEdit       write FBitsEdit;
    Property HardwareMinLabel:TLabel read  FHardwareMinLabel
                                     write FHardwareMinLabel;
    Property HardwareMaxLabel:TLabel read  FHardwareMaxLabel
                                     write FHardwareMaxLabel;
    Property UserMinLabel  : TLabel read FUserMinLabel   write FUserMinLabel;
    Property UserMaxLabel  : TLabel read FUserMaxLabel   write FUserMaxLabel;
    Property Frequency1Edit: TEdit  read FFrequency1Edit write FFrequency1Edit;
    Property Frequency0Edit: TEdit  read FFrequency0Edit write FFrequency0Edit;
    {-----Events-----}
    Property OnUserMinMaxChange: TNotifyEvent read FOnUserMinMaxChange
                                              write FOnUserMinMaxChange;
    Property OnParamError: TNotifyEvent read FOnParamError
                                        write FOnParamError;
  end;

procedure Register;




implementation




procedure Register;
begin
  RegisterComponents('Bernd', [TDdsChip]);
end;

{---------- utils ---------------}

Function TDdsChip.FreqStr(Frequency:Extended):String;
  begin
  FreqStr:=FloatToStrF(Frequency, ffFixed, 10, 3);
  end;

Function TDdsChip.Limited(Value, min, max: Extended):Extended;
  begin
  if value<min then Limited:=min else
  if value>max then Limited:=max else
    Limited:=Value;
  end;

Procedure TDdsChip.SendPhase(TC:integer);
  var
    bin: longint;
  begin
  if TC=1 then
    bin:=Round( (FFrequency1/FClock) * ZweiHochBitzahl )
  else
    bin:=Round( (FFrequency0/FClock) * ZweiHochBitzahl );
  SendBits(TC, bin, FBits);
  end;

Function TDdsChip.GetFTW(TC:integer):Longint;
  begin
  if TC=0 then
    GetFTW:=Round( (FFrequency0/FClock) * {exp(ln(2)*Bits)} ZweiHochBitzahl )
  else
    GetFTW:=Round( (FFrequency1/FClock) * {exp(ln(2)*Bits)} ZweiHochBitzahl )
  end;
{--------------------------------}

Procedure TDdsChip.ParamError;
  begin
  if assigned(FOnParamError) then FOnParamError(self);
  end;

Procedure TDdsChip.UserMinMaxChange;
  begin
  if assigned(FOnUserMinMaxChange) then FOnUserMinMaxChange(self);
  end;


constructor TDdsChip.Create(AOwner: TComponent);
  begin
  inherited Create(AOwner);
  Buffer:=$60; {= 0110 0000 binary}
  SelectedPortNumber:=0;

  ZweiHochBitzahl:=1.0;
  FQuartz:=55000.0;
  FDivMul:=Divide;
  FDivider:=1;
  FBits:=32;
  SetClockDependentVariables;
  SetUserMinMax(HardwareMin, HardwareMax);
  Frequency1:=HardwareMin;
  Frequency0:=HardwareMin;
  end;


Function TDdsChip.HardwareMin:Extended;
  begin HardwareMin:=FHardwareMin end;

Function TDdsChip.HardwareMax:Extended;
  begin HardwareMax:=FHardwareMax end;


Procedure TDdsChip.SetQuartz(F:Extended);
  begin
  F:=Limited(F, 0.001, 900000.0);
  FQuartz:=F;
  if assigned(FQuartzEdit)  then FQuartzEdit.Text:=FreqStr(FQuartz);
  SetClockDependentVariables;
  end;

Procedure TDdsChip.SetDivMul(value: tDivMul);
  begin
  if value<>FDivMul then
    begin
    FDivMul:=value;
    SetClockDependentVariables;
    end;
  end;

Procedure TDdsChip.SetDivider(D:Longint);
  begin
  FDivider:=Round(Limited(D, 1, 128)); {dont allow Zero}
  if assigned(FDividerEdit) then FDividerEdit.Text:=IntToStr(FDivider);
  SetClockDependentVariables;
  end;

Procedure TDdsChip.SetBits(B:Integer);
  begin
  b:=Round(Limited(B, 1, LastBit));
  FBits:=B;
  if assigned(FBitsEdit)    then FBitsEdit.Text:=IntToStr(FBits);
  SetClockDependentVariables;
  end;

Procedure TDdsChip.SetClockDependentVariables;
  begin
  if FDivMul=Divide then
    FClock:=FQuartz/FDivider
  else
    FClock:=FQuartz*FDivider;
  ZweiHochBitzahl:=exp(FBits*ln(2));

  if assigned(FClockLabel)  then
    FClockLabel.Caption:=FreqStr(FClock);
  FHardwareMin:=Fclock/exp(FBits*ln(2));
  FHardwareMax:=FClock*0.4;
  if assigned(FHardwareMinLabel) then
    FHardwareMinLabel.Caption:=FreqStr(FHardwareMin);
  if assigned(FHardwareMaxLabel) then
    FHardwareMaxLabel.Caption:=FreqStr(FHardwareMax);
  SetUserMinMax(FHardwareMin, FHardwareMax);
  end;


Procedure TDdsChip.SetUserMinMax(Fmin,Fmax:Extended);
  const
    epsilon=0.001;
  var
    h:Extended;
  begin
  if Fmax<Fmin then
    begin
    h:=Fmax; Fmax:=Fmin; Fmin:=h;
    end;
  if Fmax<=Fmin+epsilon then
    Fmax:=Fmin+epsilon;

  FUserMin:=Limited(FMin, FHardwareMin, FHardwareMax);
  FUserMax:=Limited(FMax, FHardwareMin, FHardwareMax);
  if assigned(FUserMinLabel) then
    FUserMinLabel.Caption:=FreqStr(FUserMin);
  if assigned(FUserMaxLabel) then
    FUserMaxLabel.Caption:=FreqStr(FUserMax);
  if assigned(FOnUserMinMaxChange) then
    UserMinMaxChange;
  Frequency1:=Limited(Frequency1, FUserMin, FUserMax);
  Frequency0:=Limited(Frequency0, FUserMin, FUserMax);
  end;

Function TDdsChip.UserMin:Extended;
  begin UserMin:=FUserMin; end;

  Function TDdsChip.UserMax:Extended;
  begin UserMax:=FUserMax; end;

Function TDdsChip.Clock: Extended;
  begin Clock:=FClock; end;

Procedure TDdsChip.SetFrequency1(F:Extended);
  begin
  FFrequency1:=Limited(F, FUserMin, FUserMax);
  SendPhase(1);
  if assigned(FFrequency1Edit) then
    FFrequency1Edit.Text:=FreqStr(FFrequency1);
  end;

Procedure TDdsChip.SetFrequency0(F:Extended);
  begin
  FFrequency0:=Limited(F, FUserMin, FUserMax);
  SendPhase(0);
  if assigned(FFrequency0Edit) then
    FFrequency0Edit.Text:=FreqStr(FFrequency0);
  end;


{parport methods}

function TDdsChip.Init_Port(PortNumber:tPortNumber):boolean;     {True if OK}
  {var Offset : word;}
  begin
  {Offset:=8+2*(PortNumber-1);
  io_adresse:=MemW[$0040:Offset];  GPF in Windows!!!}
  case PortNumber of
    1:io_adresse:=$0378;
    2:io_adresse:=$037C;
    else io_adresse:=0;
    end;
  Init_Port:=io_adresse<>0;
  end;

function TDdsChip.Test_Port(PortNumber:tPortNumber):boolean;     {True if OK}
  var Offset, ioadr : word;
  begin
  {Offset:=8+2*(PortNumber-1);
  ioadr:=MemW[$0040:Offset];}
  Test_Port:=ioadr in [1..2];
  end;

Function  TDdsChip.Port1Available:boolean; begin Port1Available:=test_port(1); end;
Function  TDdsChip.Port2Available:boolean; begin Port2Available:=test_port(2); end;


function TDdsChip.SelectPort(Number:tPortNumber):boolean;
  begin
  if init_port(Number) then
    begin
    SelectPort:=true;
    SelectedPortNumber:=Number;
    end
  else
    SelectedPortNumber:=0;
  end;


Function TDdsChip.SelectedPort:integer;
  begin
  SelectedPort:=SelectedPortNumber;
  end;



procedure TDdsChip.Port_Out_Byte(b:byte);
  begin
  if SelectedPortNumber>0 then
    begin
    Buffer:=b;
    port[io_adresse]:=Buffer;
    end;
  end;

procedure TDdsChip.Port_Set_Bit(Nummer:byte);
  begin
  if SelectedPortNumber>0 then
    begin
    if Nummer in [0..7] then
      Port_Out_Byte(Buffer or (1 shl Nummer));
    end;
  end;

procedure TDdsChip.Port_Reset_Bit(Nummer:byte);
  begin
  if SelectedPortNumber>0 then
    begin
    if Nummer in [0..7] then
      Port_Out_Byte(Buffer and not (1 shl Nummer));
    end;
  end;

function TDdsChip.Port_Get_Bit(Nummer:byte):boolean;
  begin
  if SelectedPortNumber>0 then
    begin
    if Nummer in [0..7] then
      Port_Get_Bit:=(Buffer and (1 shl Nummer)) <> 0;
    end;
  end;

procedure TDdsChip.SendBits(TC1:integer; Phase:longint; AnzahlBits:integer);
  var   i:integer;
  begin
  if AnzahlBits>32 then AnzahlBits:=32;
  case TC1 and $0001 of
    0 : Port_Reset_Bit(DDS_TC1);
    1 : Port_Set_Bit  (DDS_TC1);
    end;
  Port_Reset_Bit(DDS_Clock);
  Port_Reset_Bit(DDS_Load);
  for i:=0 to AnzahlBits-1 do
    begin
    if ((Phase shl i) and $80000000) <> 0 then Port_Set_Bit  (DDS_Data)
                                          else Port_Reset_Bit(DDS_Data);
    Port_Set_Bit(DDS_Clock);
    Port_Reset_Bit(DDS_Clock);
    end;
  Port_Set_Bit  (DDS_Load);
  Port_Reset_Bit(DDS_Load);
  end;

end.
