#!/usr/bin/perl -T
#!/usr/local/bin/perl -T

## --> Wireless Network Link Calculator, wireless.cgi
## --> Green Bay Professional Packet Radio, www.gbppr.org

## This file Copyright 2001 <contact@gbppr.org> under the GPL.
## NO WARRANTY.  Please send bug reports / patches.

# $pwr_out           Output power of transmitter in dBm
# $tx_cab_loss       Transmitter cable attenuation in dB
# $rx_cab_loss       Receiver cable attenuation in dB
# $tx_misc_loss      Misc transmitter path loss in dB, subtracted from $eirp
# $tx_misc_cab_loss  Misc transmitter cable loss
# $rx_misc_cab_loss  Misc receiver cable loss
# $con_tx            Transmit connector number
# $con_rx            Receive connector number
# $tx_ant_gain       Transmitting antenna gain in dBi
# $rx_ant_gain       Receiving antenna gain in dBi
# $pl                Free space path loss in dB
# $rx_pwr            Received power level at receiver in dBm
# $rx_sens           Receiver sensitivity in dBm
# $eirp              Effective Isotropic Radiated Power
# plus a zillion others I forgot...

# Setup
#
select STDOUT;
$| = 1;
use Math::Complex;
use Math::Trig;

my $pic = "pics/link_view.png";

# Print MIME
#
print "Content-type:text/html\n\n";

# Read environment
#
read(STDIN, $buffer, $ENV{'CONTENT_LENGTH'});
@pairs = split(/&/, $buffer);
foreach $pair (@pairs) {
  ($name, $value) = split(/=/, $pair);
  $value =~ tr/+/ /;
  $value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/eg;
  $FORM{$name} = $value;
}

my $frq = $FORM{'frq'};
my $frq_val = $FORM{'frq_val'};
my $pwr_out = $FORM{'pwr_out'};
my $pwr_out_val = $FORM{'pwr_out_val'};

my $tx_name = $FORM{'tx_name'};
my $tx_cab = $FORM{'tx_cab'};
my $tx_len = $FORM{'tx_len'};
my $tx_len_val = $FORM{'tx_len_val'};
my $tx_ant_gain = $FORM{'tx_ant_gain'};
my $tx_radome = $FORM{'tx_radome'};
my $tx_ant_val = $FORM{'tx_ant_val'};
my $tx_ant_ht = $FORM{'tx_ant_ht'};
my $tx_ant_ht_val = $FORM{'tx_ant_ht_val'};
my $tx_elv = $FORM{'tx_elv'};
my $tx_elv_val = $FORM{'tx_elv_val'};
my $con_tx = $FORM{'con_tx'};
my $tx_misc_loss = $FORM{'tx_misc_loss'};
my $tx_misc_cab_loss = $FORM{'tx_misc_cab_loss'};
my $tx_misc_gain = $FORM{'tx_misc_gain'};

my $rx_name = $FORM{'rx_name'};
my $rx_cab = $FORM{'rx_cab'};
my $rx_len = $FORM{'rx_len'};
my $rx_len_val = $FORM{'rx_len_val'};
my $rx_ant_gain = $FORM{'rx_ant_gain'};
my $rx_radome = $FORM{'rx_radome'};
my $rx_ant_val = $FORM{'rx_ant_val'};
my $rx_ant_ht = $FORM{'rx_ant_ht'};
my $rx_ant_ht_val = $FORM{'rx_ant_ht_val'};
my $rx_elv = $FORM{'rx_elv'};
my $rx_elv_val = $FORM{'rx_elv_val'};
my $con_rx = $FORM{'con_rx'};
my $rx_misc_cab_loss = $FORM{'rx_misc_cab_loss'};
my $rx_misc_gain = $FORM{'rx_misc_gain'};

my $rx_div_ant_ht = $FORM{'rx_div_ant_ht'};
my $rx_div_ant_ht_val = $FORM{'rx_div_ant_ht_val'};
my $rx_div_ant_gain = $FORM{'rx_div_ant_gain'};
my $rx_div_ant_val = $FORM{'rx_div_ant_val'};
my $rx_div_len = $FORM{'rx_div_len'};
my $rx_div_len_val = $FORM{'rx_div_len_val'};
my $rx_div_misc_cab_loss = $FORM{'rx_div_misc_cab_loss'};

my $BER = $FORM{'BER'};
my $BER_val = $FORM{'BER_val'};

my $dfm = $FORM{'dfm'};
my $eifm = $FORM{'eifm'};
my $aifm = $FORM{'aifm'};

my $nth = $FORM{'nth'};
my $k = $FORM{'k'};
my $climate = $FORM{'climate'};
my $temp = $FORM{'temp'};
my $temp_val = $FORM{'temp_val'};
my $dist = $FORM{'dist'};
my $dist_val = $FORM{'dist_val'};

my $chia = $FORM{'chia'};
my $chia1 = $FORM{'chia1'};
my $chia2 = $FORM{'chia2'};
my $chia3 = $FORM{'chia3'};
my $chia4 = $FORM{'chia4'};

my $rough = $FORM{'rough'};
my $rough_hum = $FORM{'rough_hum'};
my $rough_val = $FORM{'rough_val'};

my $land = $FORM{'land'};

my $x1 = $FORM{'x1'};
my $x2 = $FORM{'x2'};
my $y1 = $FORM{'y1'};
my $y2 = $FORM{'y2'};

my $tx_cab_other = $FORM{'tx_cab_other'};
my $rx_cab_other = $FORM{'rx_cab_other'};
my $rh = $FORM{'rh'};
my $baro = $FORM{'baro'};
my $rate = $FORM{'rate'};

# Clean up user input data & start calculations
#

# Frequency
#
$frq =~ tr/0-9.//csd;

if ($frq_val eq "GHz") {
  $frq_mhz = $frq * 1000; # convert to MHz
  $frq_ghz = $frq;
}
else {
  $frq_mhz = $frq;
  $frq_ghz = $frq / 1000;
}

if (!$frq_mhz || !$frq_ghz || $frq_mhz < 20) {
  print "<html><b>You need a enter a frequency above 20 MHz.</b></html>";
  exit;
}

# Transmitter cable type and antenna gain
#
$tx_name =~ tr/A-Za-z0-9.,-()_[]=:;'\" //csd;
$tx_len =~ tr/0-9.//csd;
$tx_ant_gain =~ tr/0-9.//csd;
$tx_radome =~ tr/0-9.//csd;
$tx_ant_ht =~ tr/0-9.//csd;
$tx_elv =~ tr/0-9.-//csd;
$con_tx =~ tr/0-9//csd;
$tx_misc_loss =~ tr/0-9.//csd;
$tx_misc_cab_loss =~ tr/0-9.//csd;
$tx_cab_other =~ tr/0-9.//csd;
$tx_misc_gain =~ tr/0-9.//csd;

if (!$tx_len) {
  print "<html><b>You need to enter the transmitter's total cable length.</b></html>";
  exit;
}

if (!$tx_ant_ht) {
  print "<html><b>You need to enter the transmitter's antenna height.</b></html>";
  exit;
}

if (!$tx_elv) {
  print "<html><b>You need to enter the transmitter's site elevation above sea level.</b></html>";
  exit;
}

# Receiver cable type and antenna gain
#
$rx_name =~ tr/A-Za-z0-9.,-()_[]=:;'\" //csd;
$rx_len =~ tr/0-9.//csd;
$rx_ant_gain =~ tr/0-9.//csd;
$rx_radome =~ tr/0-9.//csd;
$rx_ant_ht =~ tr/0-9.//csd;
$rx_elv =~ tr/0-9.-//csd;
$con_rx =~ tr/0-9//csd;
$rx_misc_cab_loss =~ tr/0-9.//csd;
$rx_cab_other =~ tr/0-9.//csd;
$rx_misc_gain =~ tr/0-9.//csd;
$rx_div_ant_ht =~ tr/0-9.//csd;
$rx_div_ant_gain =~ tr/0-9.//csd;
$rx_div_len =~ tr/0-9.//csd;
$rx_div_misc_cab_loss =~ tr/0-9.//csd;

if (!$rx_len) {
  print "<html><b>You need to enter the receiver's total cable length.</b></html>";
  exit;
}

if (!$rx_ant_ht) {
  print "<html><b>You need to enter the receiver's antenna height.</b></html>";
  exit;
}

if (!$rx_elv) {
  print "<html><b>You need to enter the receiver's site elevation above sea level.</b></html>";
  exit;
} 

# Distance
#
$dist =~ tr/0-9.//csd;

if ($chia eq "no") {
  if (!$dist) {
    print "<html><b>You need to enter the distance between the transmitter and receiver.</b></html>";
    exit;
  }
}

# Average annual temperature
#
$temp =~ tr/0-9.-//csd;

# F to Umm C & K
#
if ($temp_val eq "fahrenheit") {
  $temp_c = sprintf "%.2f", (5 / 9) * ($temp - 32);
  $temp_f = sprintf "%.2f", $temp;
  $temp_k = sprintf "%.2f", 273.15 + $temp_c;
}  
else { 
  $temp_c = sprintf "%.2f", $temp;
  $temp_f = sprintf "%.2f", ((9 / 5) * $temp) + 32;
  $temp_k = sprintf "%.2f", 273.15 + $temp_c; 
} 

# Climate & terrain factors
#
$rough =~ tr/0-9.-//csd;

if ($rough_val eq "meters") {
  $rough_m = sprintf "%.2f", $rough;
  $rough_ft = sprintf "%.2f", $rough / 0.3048;
} 
else {
  $rough_m = sprintf "%.2f", $rough * 0.3048;
  $rough_ft = sprintf "%.2f", $rough; 
} 

if ($chia4 eq "yes") {
  if ($rough_hum eq "Coastal, very humid areas") {
    $c = 2;
  } 
  elsif ($rough_hum eq "Non-coastal, humid areas") {
    $c = 1.4;
  } 
  elsif ($rough_hum eq "Average or temperate areas") {
    $c = 1;
  } 
  elsif ($rough_hum eq "Dry areas") {
    $c = 0.5;
  } 
  $cli = sprintf "%.2f", $c * (($rough_ft / 50) ** -1.3);
  $climate = "";
}
else {
  if ($climate eq "6 : Very smooth terrain, over water or flat desert, coastal") {
    $cli = sprintf "%.2f", 6;
    ($null, $climate) = split ':', $climate;
  }
  elsif ($climate eq "4 : Very smooth terrain, over water or flat desert, non-coastal") {
    $cli = sprintf "%.2f", 4;
   ($null, $climate) = split ':', $climate;
  }
  elsif ($climate eq "2 : Great lakes area") {
    $cli = sprintf "%.2f", 2;
   ($null, $climate) = split ':', $climate;
  }
  elsif ($climate eq "1 : Average terrain, with some roughness") {
    $cli = sprintf "%.2f", 1;
    ($null, $climate) = split ':', $climate;
  }
  elsif ($climate eq "0.5 : Dry desert climate") {
    $cli = sprintf "%.2f", 0.50;
    ($null, $climate) = split ':', $climate;
  }
  elsif ($climate eq "0.25 : Mountainous, very rough, very dry but non-reflective") {
    $cli = sprintf "%.2f", 0.25;
    ($null, $climate) = split ':', $climate;
  }
}

# Humidity & pressure
#
$rh =~ tr/0-9.//csd;
$baro =~ tr/0-9.//csd;

if (!$rh) {
  $rh = 50; # 50% relative humidity
} 

if (!$baro) {
  $baro = 30; # 30 inches of mercury
}

# Cordinates
#
$x1 =~ tr/0-9.-//csd;
$x2 =~ tr/0-9.-//csd;
$y1 =~ tr/0-9.-//csd;
$y2 =~ tr/0-9.-//csd;

if (!$y1 || $y1 > 90 || $y1 < -90) {
  $y1 = 44.51800;
}
if (!$x1 || $x1 > 180 || $x1 < -180) {
  $x1 = -87.96850;
}
if (!$y2 || $y2 > 90 || $y2 < -90) {
  $y2 = 44.51620;
}
if (!$x2 || $x2 > 180 || $x2 < -180) {
  $x2 = -88.02279;
}

# Transmitter RF output power in dBm
#
$pwr_out =~ tr/0-9.-//csd;

if ($pwr_out_val eq "milliwatts") {
  $pwr_out = 10 * log10($pwr_out); # mW to dBm
}
elsif ($pwr_out_val eq "watts") {
  $pwr_out = 10 * log10($pwr_out) + 30; # W to dBm
}
elsif ($pwr_out_val eq "kilowatts") {
  $pwr_out = 10 * log10($pwr_out) + 60; # kW to dBm
}
elsif ($pwr_out_val eq "dBW") {
  $pwr_out = 10 * log10((10 ** (($pwr_out + 30) / 10))); # dBW to dBm
}
elsif ($pwr_out_val eq "dBk") {
  $pwr_out = 10 * log10((10 ** (($pwr_out + 60) / 10))); # dBk to dBm
} 

$pwr_out = sprintf "%.3f", $pwr_out;
$pwr_out_mw = sprintf "%.3f", 10 ** ($pwr_out / 10);
$pwr_out_w = sprintf "%.3f", 10 ** (($pwr_out - 30) / 10);
$pwr_out_kw = sprintf "%.3f", $pwr_out_w / 1000;
$pwr_out_dbw = sprintf "%.3f", 10 * log10((10 ** (($pwr_out - 30) / 10)));
$pwr_out_dbk = sprintf "%.3f", (10 * log10($pwr_out_w)) - 30;

# Cable loss per meter
#
sub Cable {
  undef $loss_per_foot; undef $loss_per_meter;
  if ($val eq "Times Microwave LMR-400") {
    $loss_per_foot = ((0.12229 * sqrt $frq_mhz) + (0.00026 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }
  elsif ($val eq "Times Microwave LMR-400 UltraFlex") {
    $loss_per_foot = ((0.12229 * sqrt $frq_mhz) + (0.00026 * $frq_mhz)) / 100;
    $loss_per_foot = $loss_per_foot + ($loss_per_foot * 0.15);
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }
  elsif ($val eq "Times Microwave LMR-500") {
    $loss_per_foot = ((0.09659 * sqrt $frq_mhz) + (0.00026 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }
  elsif ($val eq "Times Microwave LMR-600") {
    $loss_per_foot = ((0.0755 * sqrt $frq_mhz) + (0.00026 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }
  elsif ($val eq "Times Microwave LMR-900") {
    $loss_per_foot = ((0.05177 * sqrt $frq_mhz) + (0.00016 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  } 
  elsif ($val eq "Times Microwave LMR-1200") {
    $loss_per_foot = ((0.03737 * sqrt $frq_mhz) + (0.00016 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }         
  elsif ($val eq "Times Microwave LMR-1700") {
    $loss_per_foot = ((0.02646 * sqrt $frq_mhz) + (0.00016 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }         
  elsif ($val eq "Andrew Heliax LDF4-50A") {
    $loss_per_foot = ((0.06432 * sqrt $frq_mhz) + (0.00019 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  } 
  elsif ($val eq "Andrew Heliax LDF5-50A") {
    $loss_per_foot = ((0.03482 * sqrt $frq_mhz) + (0.00015 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }        
  elsif ($val eq "Andrew Heliax LDF6-50A") {
    $loss_per_foot = ((0.02397 * sqrt $frq_mhz) + (0.00014 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }        
  elsif ($val eq "Andrew Heliax LDF7-50A") {
    $loss_per_foot = ((0.01901 * sqrt $frq_mhz) + (0.00014 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }        
  elsif ($val eq "Belden 9913 (RG-8)") {
    $loss_per_foot = ((0.12050 * sqrt $frq_mhz) + (0.00066 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }
  elsif ($val eq "Belden 8267 (RG-213)") {
    $loss_per_foot = ((0.18993 * sqrt $frq_mhz) + (0.00216 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }
  elsif ($val eq "Belden 9258 (RG-8X)") {
    $loss_per_foot = ((0.26904 * sqrt $frq_mhz) + (0.00572 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  } 
  elsif ($val eq "Belden 8240 (RG-58)") {
    $loss_per_foot = ((0.34190 * sqrt $frq_mhz) + (0.00377 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  } 
  elsif ($val eq "Crap RG-8") {
    $loss_per_foot = ((0.21 * sqrt $frq_mhz) + (0.00026 * $frq_mhz)) / 100;
    $loss_per_meter = $loss_per_foot * 3.2808399;
  }
  elsif ($val eq "Other") {
    if ($val2 == 0) {
      $val2 = 7; $val1 = "feet";
    }
    if ($val1 eq "meters") {
      $loss_per_meter = $val2 / 100;
      $loss_per_foot = $loss_per_meter / 3.2808399;
    }  
    elsif ($val1 eq "feet") {
      $loss_per_foot =  $val2 / 100;
      $loss_per_meter = $loss_per_foot * 3.2808399;
    }
  }
}

# Transmitter cable
#
&Cable($val = $tx_cab, $val1 = $chia1, $val2 = $tx_cab_other);

if ($tx_len_val eq "meters") {
  $tx_cab_loss = $tx_len * $loss_per_meter;
  $tx_length = $tx_len;
  $tx_loss_per_meter = $loss_per_meter;
  $tx_loss_per_foot = $loss_per_foot;
  $tx_loss_per_100m = $loss_per_meter * 100;
  $tx_loss_per_100f = $loss_per_foot * 100;
}
else {
  $tx_cab_loss = $tx_len * $loss_per_foot;
  $tx_length = $tx_len * 0.3048; # ft to m
  $tx_loss_per_meter = $loss_per_meter;
  $tx_loss_per_foot = $loss_per_foot;
  $tx_loss_per_100m = $loss_per_meter * 100;
  $tx_loss_per_100f = $loss_per_foot * 100;
}

# Transmit cable efficiency
#
$tx_eff = sprintf "%.2f", 100 / (10 ** ($tx_cab_loss / 10)); # percent

if ($tx_eff < 50) {
  $tx_eff_message = "<b>You should use lower loss cable</b>";
}

# Transmit connector and/or adapter loss
#
$tx_con_loss = $con_tx * (0.1 * sqrt($frq_ghz));

# Total transmitter cable loss
#
$tx_total_cable_loss = $tx_cab_loss + $tx_con_loss + $tx_misc_cab_loss;

# Receiver cable
#
&Cable($val = $rx_cab, $val1 = $chia2, $val2 = $rx_cab_other);

if ($rx_len_val eq "meters") {
  $rx_cab_loss = $rx_len * $loss_per_meter;
  $rx_length = $rx_len;
  $rx_loss_per_meter = $loss_per_meter;
  $rx_loss_per_foot = $loss_per_foot;
  $rx_loss_per_100m = $loss_per_meter * 100;
  $rx_loss_per_100f = $loss_per_foot * 100;
}
else {
  $rx_cab_loss = $rx_len * $loss_per_foot;
  $rx_length = $rx_len * 0.3048; # ft to m
  $rx_loss_per_meter = $loss_per_meter;
  $rx_loss_per_foot = $loss_per_foot;
  $rx_loss_per_100m = $loss_per_meter * 100;
  $rx_loss_per_100f = $loss_per_foot * 100;
}

# Diversity antenna line loss
#
if ($rx_div_ant_ht) {
  if ($rx_div_len_val eq "feet") {
    $rx_div_loss = ($rx_div_len * $rx_loss_per_foot) + $rx_div_misc_cab_loss;
  }
  elsif ($rx_div_len_val eq "meters") {
    $rx_div_loss = ($rx_div_len * $rx_loss_per_meter) + $rx_div_misc_cab_loss;
  }
}

# Receiver cable efficiency
#
$rx_eff = sprintf "%.2f", 100 / (10 ** ($rx_cab_loss / 10)); # percent

if ($rx_eff < 50) {
  $rx_eff_message = "<b>You should use lower loss cable</b>";
} 

# Receiver connector and/or adapter loss
#
$rx_con_loss = $con_rx * (0.1 * sqrt($frq_ghz));

# Total receiver cable loss
#
$rx_total_cable_loss = $rx_cab_loss + $rx_con_loss + $rx_misc_cab_loss;

# Antenna gain
#
if ($tx_ant_val eq "dBd") {
  $tx_ant_gain = $tx_ant_gain + 2.15; # dBi
}

if ($rx_ant_val eq "dBd") {
  $rx_ant_gain = $rx_ant_gain + 2.15;
}

if ($rx_div_ant_val eq "dBd") {
  $rx_div_ant_gain = $rx_div_ant_gain + 2.15;
} 

# Antenna 3 dB beamwidth
#
$tx_ant_bw = sprintf "%.2f", 164 * sqrt(1 / (10 ** ($tx_ant_gain / 10)));
$rx_ant_bw = sprintf "%.2f", 164 * sqrt(1 / (10 ** ($rx_ant_gain / 10)));

# Site elevation AMSL
#
if ($tx_elv_val eq "meters") {
  $tx_elv_m = sprintf "%.2f", $tx_elv;
  $tx_elv_ft = sprintf "%.2f", $tx_elv / 0.3048;
}
else {
  $tx_elv_m = sprintf "%.2f", $tx_elv * 0.3048;
  $tx_elv_ft = sprintf "%.2f", $tx_elv;
}

if ($rx_elv_val eq "meters") {
  $rx_elv_m = sprintf "%.2f", $rx_elv;
  $rx_elv_ft = sprintf "%.2f", $rx_elv / 0.3048;
} 
else {
  $rx_elv_m = sprintf "%.2f", $rx_elv * 0.3048;
  $rx_elv_ft = sprintf "%.2f", $rx_elv;
} 

# K factor
#
if ($chia3 eq "yes") {
  # Atmospheric pressure from inches of mercury to millibars with sea level 
  # correction.
  #
  $elv_ft = (sqrt($tx_elv_m * $tx_elv_m)) * 3.2808399; # m to ft
  $atmos_p = sprintf "%.2f", (33.86 * $baro) - ($elv_ft * 0.025);

  # Saturation vapor pressure, millibars
  #
  $es = sprintf "%.2f", exp(1.805 + (0.0738 * $temp_c) - (0.000298 * ($temp_c ** 2)));

  # Partial vapor pressure, millibars
  #
  $vapor_p = sprintf "%.2f", ($rh / 100) * $es;

  # Index of refraction
  #
  $N = sprintf "%.2f", ((77.6 * $atmos_p) / $temp_k) + ((373000 * $vapor_p) / $temp_k ** 2);

  # Effective Earth radius, K factor
  #
  $k = sprintf "%.2f", 1 / (1 - 0.04665 * exp(0.005577 * $N));
  
  if (!$k || $k > 10) {
    $k = sprintf "%.2f", 1.33;
  }
}
else {
  $k =~ tr/0-9.//csd;

  if (!$k || $k > 10) {
    $k = sprintf "%.2f", 1.33;
  }
  $N = $vapor_p = $es = $atmos_p = $elv_ft = "Not applicable";
}

# Radio horizon calculations
#
if ($tx_ant_ht_val eq "feet") {
  $tx_ant_ht_m = sprintf "%.2f", $tx_ant_ht * 0.3048; # feet to meters
  $tx_ant_ht_ft = sprintf "%.2f", $tx_ant_ht;
}
else {
  $tx_ant_ht_m = sprintf "%.2f", $tx_ant_ht;
  $tx_ant_ht_ft = sprintf "%.2f", $tx_ant_ht / 0.3048;
}

if ($rx_ant_ht_val eq "feet") {
  $rx_ant_ht_m = sprintf "%.2f", $rx_ant_ht * 0.3048; # feet to meters
  $rx_ant_ht_ft = sprintf "%.2f", $rx_ant_ht;
}
else {
  $rx_ant_ht_m = sprintf "%.2f", $rx_ant_ht;
  $rx_ant_ht_ft = sprintf "%.2f", $rx_ant_ht / 0.3048;
}

$tx_rad_hor = sprintf "%.2f", sqrt (12.75 * $tx_ant_ht_m * $k); # distance (km) to radio horizon
$rx_rad_hor = sprintf "%.2f", sqrt (12.75 * $rx_ant_ht_m * $k); # distance (km) to radio horizon
$tx_rad_hor_mi = sprintf "%.2f", $tx_rad_hor * 0.62137119; # distance (mi) to radio horizon
$rx_rad_hor_mi = sprintf "%.2f", $rx_rad_hor * 0.62137119; # distance (mi) to radio horizon
$tx_rad_hor_nm = sprintf "%.2f", $tx_rad_hor / 1.852; # distance (nmi) to radio horizon
$rx_rad_hor_nm = sprintf "%.2f", $rx_rad_hor / 1.852; # distance (nmi) to radio horizon

$tx_ant_ht_ov_ft = sprintf "%.2f", $tx_ant_ht_ft + $tx_elv_ft;
$tx_ant_ht_ov_m = sprintf "%.2f", $tx_ant_ht_m + $tx_elv_m;
$rx_ant_ht_ov_ft = sprintf "%.2f", $rx_ant_ht_ft + $rx_elv_ft;
$rx_ant_ht_ov_m = sprintf "%.2f", $rx_ant_ht_m + $rx_elv_m;

# Maximum communication distance
#
$distance_max = sprintf "%.2f", $tx_rad_hor + $rx_rad_hor;
$distance_max_mi = sprintf "%.2f", ($tx_rad_hor + $rx_rad_hor) * 0.62137119;
$distance_max_nm = sprintf "%.2f", $distance_max_mi / 1.1507794;

# Cordinates to distance
#
if ($chia eq "yes") {
  $YM = ($y1 + $y2) / 2;
  $H = (($x1 - $x2)) * (68.962 + 0.04525 * ($YM) - 0.01274 * ($YM ** 2) + 0.00004117 * ($YM ** 3));
  $V = (($y1 - $y2)) * (68.712 - 0.001184 * ($YM) + 0.0002928 * ($YM ** 2) - 0.000002162 * ($YM ** 3));
  $dist_mi = sprintf "%.3f", sqrt(($H ** 2) + ($V ** 2));
  $dist_nm = sprintf "%.3f", $dist_mi / 1.1507794;
  $dist_km = sprintf "%.3f", $dist_mi * 1.609344;
  $x1 = sprintf "%.7f", $x1;
  $x2 = sprintf "%.7f", $x2;
  $y1 = sprintf "%.7f", $y1;
  $y2 = sprintf "%.7f", $y2;

  sub Dec2DMS {
    undef $D; undef $M; undef $S;
    $D = int($VAL);
    $M = int(($VAL - $D) * 60);
    $S = ((($VAL - $D) * 60) - $M) * 60;
    $S = sprintf "%.1f", int($S * 1000) / 1000;
  }

  &Dec2DMS($VAL = $y1);
  $Y1 = sprintf "%.f&deg; %.f' %.2f\"", $D, abs $M, abs $S;

  &Dec2DMS($VAL = $x1);
  $X1 = sprintf "%.f&deg; %.f' %.2f\"", $D, abs $M, abs $S;

  &Dec2DMS($VAL = $y2);
  $Y2 = sprintf "%.f&deg; %.f' %.2f\"", $D, abs $M, abs $S;

  &Dec2DMS($VAL = $x2);
  $X2 = sprintf "%.f&deg; %.f' %.2f\"", $D, abs $M, abs $S;
}
else { 
  if ($dist_val eq "miles") {
    $dist_mi = sprintf "%.3f", $dist;
    $dist_nm = sprintf "%.3f", $dist_mi / 1.1507794;
    $dist_km = sprintf "%.3f", $dist * 1.609344;
  }
  else {
    $dist_km = sprintf "%.3f", $dist;
    $dist_mi = sprintf "%.3f", $dist / 1.609344;
    $dist_nm = sprintf "%.3f", $dist_mi / 1.1507794;
  }
  $x1 = $x2 = $y1 = $y2 = 0;
  $X1 = $X2 = $Y1 = $Y2 = 0;
}

# Divide by zero catches
#
if ($dist_mi == 0) {
  $dist_mi = sprintf "%.3f", 0.01;
  $dist_nm = sprintf "%.3f", $dist_mi / 1.1507794;
  $dist_km = sprintf "%.3f", $dist_mi * 1.609344;
}

if ($dist_km == 0) {
  $dist_km = sprintf "%.3f", 0.01;
  $dist_mi = sprintf "%.3f", $dist_km / 1.609344;
  $dist_nm = sprintf "%.3f", $dist_km / 1.852;
}

if ($dist_nm == 0) {
  $dist_nm = sprintf "%.3f", 0.01;
  $dist_mi = sprintf "%.3f", $dist_nm * 1.1507794;
  $dist_km = sprintf "%.3f", $dist_nm * 1.852;
}

# Bearing
#
if ($chia eq "yes") {
  $LAT1 = deg2rad $y1;
  $LNG1 = deg2rad $x1;
  $LAT2 = deg2rad $y2;
  $LNG2 = deg2rad $x2;
  $DIFF = $LNG2 - $LNG1;
  
  $CBETA = (sin($LAT1) * sin($LAT2)) + (cos($LAT1) * cos($LAT2) * cos($DIFF));
  $BETA = acos($CBETA);
  $COSAZ = (sin($LAT2) - (sin($LAT1) * cos($BETA))) / (cos($LAT1) * sin($BETA));

  if ($COSAZ > 0.999999) {
    $AZ = 0;
  }
  elsif ($COSAZ < -0.999999) {
    $AZ = 180;
  }
  else {
    $AZ = rad2deg (acos $COSAZ);
  }

  if (sin $DIFF >= 0) {
    $AZSP = $AZ;
    $AZLP = 180 + $AZ;
  }
  else {
    $AZSP = 360 - $AZ;
    $AZLP = 180 - $AZ;
  }
  $AZSP = sprintf "%.2f", $AZSP; # Azimuth from transmitter to receiver degrees East of North 
  $AZLP = sprintf "%.2f", $AZLP; # Azimuth from receiver to transmitter degrees East of North 
}
else {
  $AZSP = $AZLP = "Not applicable";
}

# 0.6 and Nth Fresnel zone outer boundry from the direct signal path
#
# Nth Fresnel zone
#
$nth =~ tr/0-9.//csd;

if (!$nth) {
  $nth = 1; # 1st Fresnel zone
} 

$half = $dist_mi / 2;
$fres_06_ft = sprintf "%.2f", 72.1 * 0.6 * sqrt($half * $half) / ($frq_ghz * $dist_mi);
$fres_n_ft = sprintf "%.2f", 72.1 * $nth * sqrt($half * $half) / ($frq_ghz * $dist_mi);

$fres_06_m = sprintf "%.2f", $fres_06_ft * 0.3048;
$fres_n_m = sprintf "%.2f", $fres_n_ft * 0.3048;

if ($nth == 0.3 || $nth == 0.4 || $nth == 0.5 || $nth == 0.7 || $nth == 0.8 || $nth == 0.9) {
 $sub = "";
}
elsif ($nth == 1) {
  $sub = "st";
} 
elsif ($nth == 2) {
  $sub = "nd";
} 
elsif ($nth == 3) {
  $sub = "rd";
} 
else {
  $sub = "th";
}

# Wavelength
#
$wav = (2.99792458E+12 / $frq_mhz) / 1E+10; # wavelength in meters
$wav_in = sprintf "%.2f", ($wav * 100) / 2.54; # inches
$wav_ft = sprintf "%.2f", $wav / 0.3048; # feet
$wav_cm = sprintf "%.2f", $wav * 100; # centimeters
$wav_m = sprintf "%.2f", $wav; # meters

# Atmospheric attenuation (oxygen loss)
#
if ($frq_ghz >= 9) {
  $oxy_att_km = sprintf "%.3f", ((6.6 / (($frq_ghz ** 2) + 0.33)) + (9 / (($frq_ghz - 57) ** 2) + 1.96)) * ($frq_ghz ** 2) * (10 ** -4);
  $oxy_att_mi = sprintf "%.3f", $oxy_att_km / 1.609344;
  $oxy_att_total = sprintf "%.3f", $oxy_att_km * $dist_km;
}
else {
  $oxy_att_km = sprintf "%.3f", 0; 
  $oxy_att_mi = sprintf "%.3f", 0; 
  $oxy_att_total = sprintf "%.3f", 0;
}
# Water vapor attenuation
#
if ($frq_ghz >= 2) {
  $wvd = 7.5; # water vapor density, gm/m3
  $water_att_km = sprintf "%.3f", ((0.067 + 2.4) / ((($frq_ghz - 22.3) ** 2) + 6.6)) * (($frq_ghz ** 2) * $wvd * (10 ** -4));
  $water_att_mi = sprintf "%.3f", $water_att_km / 1.609344;
  $water_att_total = sprintf "%.3f", $water_att_km * $dist_km;
}
else {
  $water_att_km = sprintf "%.3f", 0;
  $water_att_mi = sprintf "%.3f", 0;
  $water_att_total = sprintf "%.3f", 0;
}

# Rain attenuation
#
if (!$rate) {
  $rain_att_km = sprintf "%.3f", 0;
  $rain_att_mi = sprintf "%.3f", 0;
  $rain_att_total = sprintf "%.3f", 0;
}
else {
  $rate =~ tr/0-9.//csd;
  $rate = sprintf "%.2f", $rate;

  # Ryde & Ryde
  #
  if ($frq_ghz >= 2) {
    if ($frq_ghz >= 2 && $frq_ghz <= 54) {
      $af = 4.21 * (10 ** -5) * ($frq_ghz ** 2.42);
    }
    elsif ($frq_ghz >= 54 && $frq_ghz <= 180) {
      $af = 4.09 * (10 ** -2) * ($frq_ghz ** 0.699);
    } 
    if ($frq_ghz >= 2 && $frq_ghz <= 25) {
      $bf = 1.41 * ($frq_ghz ** -0.0779);
    }
    elsif ($frq_ghz >= 25 && $frq_ghz <= 180) {
      $bf = 2.63 * ($frq_ghz ** -0.272);
    } 
    $rain_att_km = sprintf "%.3f", $af * ($rate ** $bf); # dB/km
    $rain_att_mi = sprintf "%.3f", $rain_att_km / 1.609344;
    $rain_att_total = sprintf "%.3f", $rain_att_km * ($dist_km / (1 + (0.045 * $dist_km)));
  } 
  else {
    $rain_att_km = sprintf "%.3f", 0;
    $rain_att_mi = sprintf "%.3f", 0;
    $rain_att_total = sprintf "%.3f", 0;
  } 
}

# Free space path loss
#
$fs = sprintf "%.2f", (20 * (log10 $frq_mhz)) + (20 * (log10 $dist_km)) + 32.447782;

# Total path loss
#
$pl = sprintf "%.2f", $fs + $rain_att_total + $water_att_total + $oxy_att_total;

# Effective Isotropic Radiated Power (EIRP)
#
$eirp = sprintf "%.3f", ($pwr_out - $tx_total_cable_loss) + $tx_misc_gain + ($tx_ant_gain - $tx_radome);
$eirp_mw = sprintf "%.3f", 10 ** ($eirp / 10);
$eirp_w = sprintf "%.3f", 10 ** (($eirp - 30) / 10);
$eirp_dbw = sprintf "%.3f", 10 * log10((10 ** (($eirp - 30) / 10)));
$eirp_kw = sprintf "%.3f", $eirp_w / 1000;
$eirp_dbk = sprintf "%.3f", (10 * log10($eirp_w)) - 30;

if ($eirp > 36) {
  $pwr_message = "<b>Violates FCC Part 15.247 rules for multipoint links.</b>";
}

# Electric field intensity
#
$dbu = sprintf "%.2f", $eirp_dbk - (20 * log10($dist_km)) + 106.92; # 50-ohm system

# RF input to the antenna
#
$tx_ant_input_mw = sprintf "%.2f", 10 ** ((($pwr_out - $tx_total_cable_loss) + $tx_misc_gain) / 10);
$tx_ant_input = sprintf "%.2f", ($pwr_out - $tx_total_cable_loss) + $tx_misc_gain;

# Maximum allowed antenna RF input power, FCC Part 15.247
#
if ($tx_ant_gain > 6) {
  $max_ant_pwr = sprintf "%.2f", 30 - (($tx_ant_gain - 6) / 3);
  $max_ant_pwr_mw = sprintf "%.2f", 10 ** ($max_ant_pwr / 10);
}
else {
  $max_ant_pwr = sprintf "%.2f", 30;
  $max_ant_pwr = sprintf "%.2f", 10 ** (30 / 10);
}

# Maximum allowed antenna gain, FCC Part 15.247
#
#$max_ant_gain = sprintf "%.2f", (($max_ant_pwr - 30) * -3) + 6;
#$max_ant_gain_dbd = sprintf "%.2f", $max_ant_gain - 2.15;

# Received power level
#
$rx_pwr = sprintf "%.2f", ($eirp - ($pl + $tx_misc_loss)) + ((($rx_ant_gain - $rx_radome) + $rx_misc_gain) - $rx_total_cable_loss);
$rx_div_pwr = sprintf "%.2f", ($eirp - ($pl + $tx_misc_loss)) + ($rx_div_ant_gain - $rx_div_loss);
$rx_uvolt = sprintf "%.2f", (sqrt(( 10 ** ($rx_pwr / 10) / 1000) * 50)) * 1000000;

# Receiver threshold
#
$BER =~ tr/0-9.-//csd;

if (!$BER) {
  $BER = -77; $BER_val = "dBm"; # receiver threshold, in negative dBm
}

if ($BER_val eq "microvolts") {
  $BER_dbm = sprintf "%.2f", 10 * (log((($BER / 1000000) ** 2 / 50) * 1000) * 0.43429); # uV to dBm
  $BER_uvolt = sprintf "%.2f", $BER; # uV
}
else {
  $BER_dbm = sprintf "%.2f", $BER; # dBm
  $BER_uvolt = sprintf "%.2f", (sqrt((10 ** ($BER / 10) / 1000) * 50)) * 1000000; # dBm to uV
}

# Minimum antenna height
#
$min_ht_ft = sprintf "%.2f", 43.24 * sqrt($dist_mi / (4 * $frq_ghz)) +  (($dist_mi ** 2) / 8);
$min_ht_m = sprintf "%.2f", $min_ht_ft * 0.3048;

# Thermal noise fade margin
#
if ($rx_pwr >= $BER_dbm) {
  $tfm = sprintf "%.2f", abs ($BER_dbm - $rx_pwr);
}
else {
  $tfm = sprintf "%.2f", ($BER_dbm - $rx_pwr) * -1;
}

if ($rx_div_ant_ht) {
  if ($rx_div_pwr >= $BER_dbm) {
    $div_tfm = sprintf "%.2f", abs ($BER_dbm - $rx_div_pwr);
  } 
  else {
    $div_tfm = sprintf "%.2f", ($BER_dbm - $rx_div_pwr) * -1;
  } 
}
else {
  $div_tfm = "Not applicable";
}

# Composite, Dispersive, external interference, adjacent channel fade margins
#
$dfm =~ tr/0-9.-//csd;
$eifm =~ tr/0-9.-//csd;
$aifm =~ tr/0-9.-//csd;

if (!$dfm) {
  $dfm = $comp = $eifm = $aifm = "Not applicable";
}
elsif ($dfm && !$eifm && !$aifm) {
  $comp = -10 * log10(((10 ** (-$dfm / 10))) + ((10 ** (-$tfm / 10))));
  $comp = sprintf "%.2f", $comp;
  $dfm = sprintf "%.2f", $dfm;
  $eifm = $aifm = "Not applicable";
}
elsif ($dfm && $eifm && !$aifm) {
  $comp = -10 * log10(((10 ** (-$dfm / 10))) + ((10 ** (-$tfm / 10))) + ((10 ** (-$eifm / 10))));
  $comp = sprintf "%.2f", $comp;
  $dfm = sprintf "%.2f", $dfm;
  $eifm = sprintf "%.2f", $eifm;
  $aifm = "Not applicable";
}
elsif ($dfm && $aifm && !$eifm) {
  $comp = -10 * log10(((10 ** (-$dfm / 10))) + ((10 ** (-$tfm / 10))) + ((10 ** (-$aifm / 10))));
  $comp = sprintf "%.2f", $comp;
  $dfm = sprintf "%.2f", $dfm;
  $aifm = sprintf "%.2f", $aifm;
  $eifm = "Not applicable";
}
else {
  $comp = -10 * log10(((10 ** (-$dfm / 10))) + ((10 ** (-$tfm / 10))) + ((10 ** (-$eifm / 10))) + ((10 ** (-$aifm / 10))));
  $comp = sprintf "%.2f", $comp;
  $dfm = sprintf "%.2f", $dfm;
  $eifm = sprintf "%.2f", $eifm;
  $aifm = sprintf "%.2f", $aifm;
}

if ($comp eq "Not applicable") {
  $cmp = $tfm;
}
else { 
  $cmp = $comp;
}

# North American outage calculations, Vigants
#

# Annual multipath outage probability, non-diversity
#
$Und_nodiv = $cli * ($frq_ghz / 4) * (10 ** -5) * ($dist_mi ** 3) * (10 ** (-$cmp / 10));
$Und_nodiv_per = sprintf "%.9f", 100 * (1 - $Und_nodiv);
$Und_nodiv_out = sprintf "%.9f", 100 - $Und_nodiv - $Und_nodiv_per;
$SES_nodiv_mo = sprintf "%.3f", $Und_nodiv * 2600000; # SES per month (2600000 seconds)
$SES_nodiv_yr = sprintf "%.3f", $SES_nodiv_mo * 3 * ($temp_f / 50); # SES per year over a 3-month fade season

if ($SES_nodiv_mo >= 3600) {
  $worst_nodiv_mo = sprintf "%.3f", $SES_nodiv_mo / 3600;
  $worst_nodiv_mo_val = "hours";
}
elsif ($SES_nodiv_mo >= 60) {
  $worst_nodiv_mo = sprintf "%.3f", $SES_nodiv_mo / 60;
  $worst_nodiv_mo_val = "minutes";
}
else {
  $worst_nodiv_mo = sprintf "%.3f", $worst_nodiv_mo;
  $worst_nodiv_mo_val = "seconds";
}

# Diversity antenna spacing
#
if (!$rx_div_ant_ht) {
  $div_m = sprintf "%.2f", $wav_m * ((3 * ($dist_km * 1000)) / (8 * $tx_ant_ht_m));
  $div_ft = sprintf "%.2f", $div_m / 0.3048;
  $div_message = "(calculated)";
}
else {
  if ($rx_div_ant_ht_val eq "feet") {
    $div_ft = $rx_div_ant_ht;
    $div_m = $rx_div_ant_ht * 0.3048;
  }
  elsif ($rx_div_ant_ht_val eq "meters") {
    $div_m = $rx_div_ant_ht;
    $div_ft = $rx_div_ant_ht / 0.3048;
  }
  $div_ft = sprintf "%.2f", abs($div_ft - $rx_ant_ht_ft);
  $div_message = "(provided)";
}

# Space diversity improvement factor for vertically separated 
# receive antennas, Hosoya
#
$div_factor = sprintf "%.2f", 7 * (10 ** -5) * $frq_ghz * ($div_ft ** 2) * (10 ** ($div_tfm / 10)) / $dist_mi;

if ($div_factor < 1) {
  $div_fact_message = "<b>This space diversity setup will not improve link reliability</b>";
}

# Divide by zero catch
#
if ($div_factor == 0) {
  $div_factor = 0.1;
}

# Annual multipath outage probability, with diversity
#
$Und_div = $Und_nodiv / $div_factor;
$Und_div_per = sprintf "%.9f", 100 * (1 - $Und_div);
$Und_div_out = sprintf "%.9f", 100 - $Und_div - $Und_div_per;
$SES_div_mo = sprintf "%.3f", $Und_div * 2600000; # SES per month (2600000 seconds)
$SES_div_yr = sprintf "%.3f", $SES_div_mo * 3 * ($temp_f / 50); # SES per year over a 3-month fade season

if ($SES_div_mo >= 3600) {
  $worst_div_mo = sprintf "%.3f", $SES_div_mo / 3600;
  $worst_div_mo_val = "hours";
}
elsif ($SES_div_mo >= 60) {
  $worst_div_mo = sprintf "%.3f", $SES_div_mo / 60;
  $worst_div_mo_val = "minutes";
}
else {
  $worst_div_mo = sprintf "%.3f", $worst_div_mo;
  $worst_div_mo_val = "seconds";
}

# Ideal fade margin
#
$min_fade = log10(((0.9995 - 1) / (-2.5 * 2 * $cli * $frq_ghz * ($dist_mi ** 3) * (10 ** -6)))) * -10;
$min_fade = sprintf "%.2f", abs $min_fade;

# Impossible link
#
if ($tfm <= 0) {
  $chalupa = 1000;;
  $fade_message = "<b>Current link setup will not work</b>";
  $Und_div_per = $Und_nodiv_per = 0;
  $Und_div_out = $Und_nodiv_out = 100;
  $worst_div_mo = $worst_nodiv_mo = 43333;
  $worst_div_mo_val = $worst_nodiv_mo_val = "minutes";
  $SES_div_yr = $SES_nodiv_yr = "31557600";
  $SES_div_mo = $SES_nodiv_mo = "2600000";
}
elsif ($tfm > 0 && $tfm <= 10) {
  $chalupa = int(rand 10) + 10;
  $fade_message = "<b>Very low fade margin</b>";
}
elsif ($tfm > 10 && $tfm <= 20) {
  $chalupa = int(rand 10) + 1;
  $fade_message = "<b>Low fade margin</b>";
}
elsif ($tfm > 20 && $tfm <= 30) {
  $chalupa = int(rand 10) + 1;
  $fade_message = "<b>Medium fade margin</b>";
}
elsif ($tfm > 30 && $tfm <= 40) {
  $chalupa = int(rand 10) + 1; 
  $fade_message = "<b>Strong fade margin</b>";
}
elsif ($tfm > 40 && $tfm <= 50) {
  $chalupa = int(rand 10) + 1; 
  $fade_message = "<b>Very strong fade margin</b>";
}
else {
  $chalupa = 1;
  $fade_message = "<b>Extremely strong fade margin</b>";
}

# Foliage loss, per meter
#
if ($frq_ghz < 3) {
  $foli_m = sprintf "%.2f", 0.25 * ($frq_ghz ** 0.77);
  $foli_ft = sprintf "%.2f", $foli_m * 0.3048;
}
else {
  $foli_m = sprintf "%.2f", 1.102 + (1.44 * log10($frq_ghz));
  $foli_ft = sprintf "%.2f", $foli_m * 0.3048;
}

# Antenna tilt angles
#
$tilt_tr = (180 / pi) * ((($rx_ant_ht_ov_ft - $tx_ant_ht_ov_ft) / (5280.0 * $dist_mi)) - ($dist_mi / (7920.0 * $k)));
$tilt_rt = (180 / pi) * ((($tx_ant_ht_ov_ft - $rx_ant_ht_ov_ft) / (5280.0 * $dist_mi)) - ($dist_mi / (7920.0 * $k)));
$tilt_tr = sprintf "%.2f", $tilt_tr;
$tilt_rt = sprintf "%.2f", $tilt_rt;

($tilt_tr < 0) ? ($dir_tr = "(downward)") : ($dir_tr = "(upward)"); 
($tilt_rt < 0) ? ($dir_rt = "(downward)") : ($dir_rt = "(upward)");

# Urban environment loss
#
$land =~ tr/0-9.//csd;

$urban = sprintf "%.2f", 40 * log10(($dist_km * 1000)) - 20 * log10($tx_ant_ht_m - $rx_ant_ht_m) + 20 + ($frq_mhz / 40) + (1.08 * $land) - (0.34 * $rough_m);
$land = sprintf "%.2f", $land;

# Make all pretty
#
$frq_ghz = sprintf "%.6f", $frq_ghz;
$frq_mhz = sprintf "%.6f", $frq_mhz;

$tx_loss_per_foot = sprintf "%.2f", $tx_loss_per_foot;
$tx_loss_per_meter = sprintf "%.2f", $tx_loss_per_meter;
$tx_loss_per_100f = sprintf "%.2f", $tx_loss_per_100f;
$tx_loss_per_100m = sprintf "%.2f", $tx_loss_per_100m;
$rx_loss_per_foot = sprintf "%.2f", $rx_loss_per_foot;
$rx_loss_per_meter = sprintf "%.2f", $rx_loss_per_meter;
$rx_loss_per_100f = sprintf "%.2f", $rx_loss_per_100f;
$rx_loss_per_100m = sprintf "%.2f", $rx_loss_per_100m;

$con_tx = sprintf "%1.0f", $con_tx;
$tx_length_ft = sprintf "%.2f", $tx_length / 0.3048;
$tx_length = sprintf "%.2f", $tx_length;
$tx_con_loss = sprintf "%.2f", $tx_con_loss;
$tx_cab_loss = sprintf "%.2f", $tx_cab_loss;
$tx_ant_gain = sprintf "%.2f", $tx_ant_gain;
$tx_ant_gain_dbd = sprintf "%.2f", $tx_ant_gain - 2.15;
$tx_radome = sprintf "%.2f", $tx_radome;
$tx_misc_loss = sprintf "%.2f", $tx_misc_loss;
$tx_misc_cab_loss = sprintf "%.2f", $tx_misc_cab_loss;
$tx_total_cable_loss = sprintf "%.2f", $tx_total_cable_loss;
$tx_misc_gain = sprintf "%.2f", $tx_misc_gain;

$con_rx = sprintf "%1.0f", $con_rx;
$rx_length_ft = sprintf "%.2f", $rx_length / 0.3048;
$rx_length = sprintf "%.2f", $rx_length;
$rx_con_loss = sprintf "%.2f", $rx_con_loss;
$rx_cab_loss = sprintf "%.2f", $rx_cab_loss;
$rx_ant_gain = sprintf "%.2f", $rx_ant_gain;
$rx_ant_gain_dbd = sprintf "%.2f", $rx_ant_gain - 2.15;
$rx_radome = sprintf "%.2f", $rx_radome;
$rx_misc_cab_loss = sprintf "%.2f", $rx_misc_cab_loss;
$rx_total_cable_loss = sprintf "%.2f", $rx_total_cable_loss;
$rx_misc_gain = sprintf "%.2f", $rx_misc_gain;

$date = scalar gmtime;

# Draw me a web page
#
$b = "<font color=\"blue\">";
$r = "<font color=\"red\">";
$p = "<font color=\"purple\">";
$e = "</font>";

print <<EOF;
<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 3.2//EN">
<html>
<head>
<title>Wireless Network Link Analysis Results</title>
</head>
<body bgcolor="#D3D3D3" text="#000000" link="blue">
<center>
<h2>Wireless Network Link Analysis Results</h2>
</center>

<hr noshade>
<center><img src="$pic" alt="[link view]"></center>
<hr noshade>

<pre>
                       $b Highest transmitted frequency : $e$frq_mhz$b MHz ($e$frq_ghz$b GHz) $e
                       $b Transmit frequency wavelength : $e$wav_m$b meters ($e$wav_cm$b centimeters) ($e$wav_ft$b feet) ($e$wav_in$b inches) $e
                         $b Transmitter RF power output : $e$pwr_out$b dBm ($e$pwr_out_mw$b milliwatts) $e
					             $b : $e$pwr_out_dbw$b dBW ($e$pwr_out_w$b watts) $e
                                                     $b : $e$pwr_out_dbk$b dBk ($e$pwr_out_kw$b kilowatts) $e 
<hr>
                              $b Transmitter cable type : $e$tx_cab
                            $b Transmitter cable length : $e$tx_length$b meters ($e$tx_length_ft$b feet) $e
                $b Transmitter cable loss specification : $e$tx_loss_per_100m$b dB/100 meters ($e$tx_loss_per_100f$b dB/100 feet) $e
                   $b Calculated transmitter cable loss : $e$tx_cab_loss$b dB ($e$tx_loss_per_meter$b dB/meter) ($e$tx_loss_per_foot$b dB/foot) $e
		        $b Transmitter cable efficiency : $e$tx_eff$b % $e $tx_eff_message
                    $b Total transmitter connector loss : $e$tx_con_loss$b dB through $e$con_tx$b connectors $e
                $b Transmitter cable miscellaneous loss : $e$tx_misc_cab_loss$b dB $e
                        $b Total transmitter cable loss : $e$tx_total_cable_loss$b dB $e
<br>
               $b Transmitter path miscellaneous losses : $e$tx_misc_loss$b dB $e
		      $b Transmitter miscellaneous gain : $e$tx_misc_gain$b dB $e
                     $b Transmitter antenna's peak gain : $e$tx_ant_gain$b dBi ($e$tx_ant_gain_dbd$b dBd) $e
                   $b Transmitter antenna's radome loss : $e$tx_radome$b dB $e
   $b Transmitter antenna's 3 dB (half-power) beamwidth : $e$tx_ant_bw$b degrees $e
                 $b Total RF input power to the antenna : $e$tx_ant_input$b dBm ($e$tx_ant_input_mw$b milliwatts) $e 
       $b Part 15.247 allowed RF input power to antenna : $e$max_ant_pwr$b dBm ($e$max_ant_pwr_mw$b milliwatts) $e
                  $b Transmitter antenna's height (AGL) : $e$tx_ant_ht_m$b meters ($e$tx_ant_ht_ft$b feet) (center-of-radation) $e
         $b Transmitter antenna's site elevation (AMSL) : $e$tx_elv_m$b meters ($e$tx_elv_ft$b feet) $e
         $b Overall transmitter antenna's height (AMSL) : $e$tx_ant_ht_ov_m$b meters ($e$tx_ant_ht_ov_ft$b feet) $e
 $b Distance to the radio horizon from transmitter site : $e$tx_rad_hor$b kilometers ($e$tx_rad_hor_mi$b statute miles) ($e$tx_rad_hor_nm$b nautical miles) $e
     $b Transmitter to receiver antenna mechanical tilt : $e$tilt_tr$b degrees $dir_tr $e
</pre>
<hr>
<pre>
                                 $r Receiver cable type : $e$rx_cab
                               $r Receiver cable length : $e$rx_length$r meters ($e$rx_length_ft$r feet) $e
                   $r Receiver cable loss specification : $e$rx_loss_per_100m$r dB/100 meters ($e$rx_loss_per_100f$r dB/100 feet) $e
                      $r Calculated receiver cable loss : $e$rx_cab_loss$r dB ($e$rx_loss_per_meter$r dB/meter) ($e$rx_loss_per_foot$r dB/foot) $e
		           $r Receiver cable efficiency : $e$rx_eff$r % $e $rx_eff_message
                       $r Total receiver connector loss : $e$rx_con_loss$r dB through $e$con_rx$r connectors $e
                   $r Receiver cable miscellaneous loss : $e$rx_misc_cab_loss$r dB $e
                           $r Total receiver cable loss : $e$rx_total_cable_loss$r dB $e
<br>
                         $r Receiver miscellaneous gain : $e$rx_misc_gain$r dB $e
                        $r Receiver antenna's peak gain : $e$rx_ant_gain$r dBi ($e$rx_ant_gain_dbd$r dBd) $e
                      $r Receiver antenna's radome loss : $e$rx_radome$r dB $e
      $r Receiver antenna's 3 dB (half-power) beamwidth : $e$rx_ant_bw$r degrees $e
 $r Receiver antenna's center-of-radiation height (AGL) : $e$rx_ant_ht_m$r meters ($e$rx_ant_ht_ft$r feet) $e
            $r Receiver antenna's site elevation (AMSL) : $e$rx_elv_m$r meters ($e$rx_elv_ft$r feet) $e
            $r Overall receiver antenna's height (AMSL) : $e$rx_ant_ht_ov_m$r meters ($e$rx_ant_ht_ov_ft$r feet) $e
    $r Distance to the radio horizon from receiver site : $e$rx_rad_hor$r kilometers ($e$rx_rad_hor_mi$r statute miles) ($e$rx_rad_hor_nm$r nautical miles) $e
     $r Receiver to transmitter antenna mechanical tilt : $e$tilt_rt$r degrees $dir_rt $e
</pre>
<hr>
<pre>
			     $p Transmitter site's name : $e$tx_name
                         $p Transmitter site's latitude : $e$y1$p ($e$Y1$p) longitude : $e$x1$p ($e$X1$p) $e
			        $p Receiver site's name : $e$rx_name
                            $p Receiver site's latitude : $e$y2$p ($e$Y2$p) longitude : $e$x2$p ($e$X2$p) $e
  $p Azimuth from the transmitter site to receiver site : $e$AZSP$p degrees off true North $e
  $p Azimuth from the receiver site to transmitter site : $e$AZLP$p degrees off true North $e
                                 $p Total path distance : $e$dist_km$p kilometers ($e$dist_mi$p statute miles) ($e$dist_nm$p nautical miles) $e
<br>
                          $p Total free space path loss : $e$fs$p dBi $e
                 $p Total worst-case precipitation loss : $e$rain_att_total$p dB $e
                              $p Total water vapor loss : $e$water_att_total$p dB $e
                                   $p Total oxygen loss : $e$oxy_att_total$p dB $e
                              $p Total system path loss : $e$pl$p dBi $e
<br>
      $p Peak effective isotropic radiated power (EIRP) : $e$eirp$p dBm ($e$eirp_mw$p milliwatts) $e $pwr_message
						     $p : $e$eirp_dbw$p dBW ($e$eirp_w$p W) $e 
						     $p : $e$eirp_dbk$p dBk ($e$eirp_kw$p kW) $e
		            $p Electric field intensity : $e$dbu$p dB&micro $e
                $p Unfaded received carrier power level : $e$rx_pwr$p dBm ($e$rx_uvolt$p &micro;V) $e
                  $p Receiver's threshold (sensitivity) : $e$BER_dbm$p dBm ($e$BER_uvolt$p &micro;V) $e
                    $p Thermal noise (flat) fade margin : $e$tfm$p dB $e $fade_message
          $p Diversity thermal noise (flat) fade margin : $e$div_tfm$p dB $e
                              $p Dispersive fade margin : $e$dfm$p dB $e
                   $p External interference fade margin : $e$eifm$p dB $e
           $p Adjacent channel interference fade margin : $e$aifm$p dB $e
                               $p Composite fade margin : $e$comp$p dB $e
$p Ideal thermal/composite fade margin for this climate : $e$min_fade$p dB $e
<br>
  $p Dense, dry, in-leaf temperate climate foliage loss : $e$foli_m$p dB/meter ($e$foli_ft$p dB/foot) (worst case) $e
          $p Estimated attenuation due to precipitation : $e$rain_att_km$p dB/km ($e$rain_att_mi$p dB/mi) ($e$rate$p mm/hour) $e
 $p Estimated attenuation due to water vapor, $wvd gm/m<sup>3</sup> : $e$water_att_km$p dB/km ($e$water_att_mi$p dB/mi) $e
           $p  Estimated attenuation due to oxygen loss : $e$oxy_att_km$p dB/km ($e$oxy_att_mi$p dB/mi) $e
               $p Estimated urban environment path loss : $e$urban$p dB ($e$land$p % land usage) $e
  $p Absolute minimum antenna height for either antenna : $e$min_ht_m$p meters ($e$min_ht_ft$p feet) (over flat terrain) $e 
<br>
               $p Annual multipath reliability estimate : $e$Und_nodiv_per$p % (1-way, no space diversity) $e
                    $p Annual multipath outage estimate : $e$Und_nodiv_out$p % (1-way, no space diversity) $e
                        $p Worst month multipath outage : $e$worst_nodiv_mo$p $worst_nodiv_mo_val (1-way, no space diversity) $e
                      $p Severely errored seconds (SES) : $e$SES_nodiv_yr$p per year (1-way, no space diversity) $e
                      $p Severely errored seconds (SES) : $e$SES_nodiv_mo$p worst month (1-way, no space diversity) $e
<br>
             $p Vertical spacing for diversity antennas : $e$div_m$p meters ($e$div_ft$p feet) $div_message $e
	                $p Diversity improvement factor : $e$div_factor  $div_fact_message
               $p Annual multipath reliability estimate : $e$Und_div_per$p % (1-way, with space diversity) $e
                    $p Annual multipath outage estimate : $e$Und_div_out$p % (1-way, with space diversity) $e
                        $p Worst month multipath outage : $e$worst_div_mo$p $worst_nodiv_mo_val (1-way, with space diversity) $e
                      $p Severely errored seconds (SES) : $e$SES_div_yr$p per year (1-way, with space diversity) $e
                      $p Severely errored seconds (SES) : $e$SES_div_mo$p worst month (1-way, with space diversity) $e
<br>
      $p Link path's midpoint 0.6 Fresnel zone boundary : $e$fres_06_m$p meters ($e$fres_06_ft$p feet) $e
      $p Link path's midpoint $nth$sub Fresnel zone boundary : $e$fres_n_m$p meters ($e$fres_n_ft$p feet) $e
                          $p Effective Earth radius (K) : $e$k
                                      $p Climate factor : $e$cli$p $climate $e
$p Terrain roughness (standard deviation of elevations) : $e$rough_m$p meters ($e$rough_ft$p feet) $e
                          $p Average annual temperature : $e$temp_c$p degrees C ($e$temp_f$p degrees F) ($e$temp_k$p K) $e
		                $p Atmospheric pressure : $e$atmos_p$p millibars (sea level corrected) $e
		           $p Saturation vapor pressure : $e$es$p millibars $e
		              $p Partial vapor pressure : $e$vapor_p$p millibars $e
		                 $p Index of refraction : $e$N$p N units $e
          $p Maximum space wave communications distance : $e$distance_max$p kilometers ($e$distance_max_mi$p statute miles) ($e$distance_max_nm$p nautical miles) $e
         $p Number of <a href="http://www.tacobell.com/2product/2menu/chalupas.htm">Chalupas</a> to get this link working : $e$chalupa
</pre>

<p><b><u>Maps</u></b></p>

<p>Direct links to the following maps, if you entered the correct latitudes/longitudes.</p>

<blockquote>
<p>Transmitter site's <a href="http://terraserver.homeadvisor.msn.com/image.asp?Lat=$y1&Lon=$x1">US Geological Survey photo</a>, <a href="http://www.topozone.com/map.asp?lat=$y1&lon=$x1">topographical map</a> and <a href="http://www.mapblast.com/myblast/map.mb?CMD=LFILL&CT=$y1%3A$x1%3A20000&IC=&LV=&GMI=&GAD1=&GAD2=&GAD3=&GAD4=&AD2=&noPrefs=&req_action=crmap&skip=&serch=adv&PHONE=&noBRP=&remLoc=&AD4=USA&AD2_street=&AD3=&apmenu=&apcode=&areaCode=&first3=&eLat=$y1&eLon=$x1&lkup=&selCategory=&x=35&y=15">street map</a></p>

<p>Receiver site's <a href="http://terraserver.homeadvisor.msn.com/image.asp?Lat=$y2&Lon=$x2">US Geological Survey photo</a>, <a href="http://www.topozone.com/map.asp?lat=$y2&lon=$x2">topographical map</a> and <a href="http://www.mapblast.com/myblast/map.mb?CMD=LFILL&CT=$y2%3A$x2%3A20000&IC=&LV=&GMI=&GAD1=&GAD2=&GAD3=&GAD4=&AD2=&noPrefs=&req_action=crmap&skip=&serch=adv&PHONE=&noBRP=&remLoc=&AD4=USA&AD2_street=&AD3=&apmenu=&apcode=&areaCode=&first3=&eLat=$y2&eLon=$x2&lkup=&selCategory=&x=35&y=15">street map</a></p>
</blockquote>

<hr noshade size="5">

<blockquote>
<ul type="circle">
<li>Annual outage time due to multipath fade activity in a microwave link is computed over a one-year period.&nbsp;&nbsp;Actual outage is defined as that occuring over a 3-month &quot;fade season&quot;.</li>

<li>You only need to take into account Earth curvature on paths longer than approximately 11 kilometers (7 miles).</li>

<li>All cables, antennas, connectors, and adapters have a 50 ohm impedance.</li>

<li>Receive and threshold voltages are for a 50 ohm load impedance and don't take into account the receiver's bandwidth or temperature.</li>

<li>Horizontal polarization will generally provide less multipath in urban areas and may provide lower path loss in non line-of-sight situations.&nbsp;&nbsp;It is also better in reducing foliage attenuation.</li>

<li>Neither space diversity nor in-band frequency diversity provides any improvement against rain attenuation.</li>

<li>For digital radio links much smaller diversity spacing distances can be used. (e.g. only 1/7 the calculated value)</li>

<li>Space diversity is needed when crossing flat, wet surfaces.</li>

<li>Vertically polarized high-frequency link rain outage is 40-60% less than those links horizontally polarized.</li>

<li>Rain attenuation is based on formulas by Ryde &amp; Ryde</li>

<li>Principal atmospheric absorption is by oxygen and water vapor.&nbsp;&nbsp;The attenuation due to oxygen is relatively linear in the 2 to 14 GHz frequency range.&nbsp;&nbsp;Water vapor absorption is highly dependent on the frequency, as well as to the density of the water vapor (absolute humidity).</li>

<li>Attenuation from trees is approximately 0.35 dB/meter at 2.4 GHz.&nbsp;&nbsp;At lower frequencies, the attenuation is somewhat lower for horizontal polarization than for vertical, but the difference disappears above about 1 GHz.</li>

<li>Urban environment path loss is the estimated loss as the result of terrain height differences and any type of buildings in the radio path.&nbsp;&nbsp;The loss is for reference only as it's very difficult to predict accurately.&nbsp;&nbsp;It's basically the path loss you'd have if you didn't have line-of-sight and had to shoot a path through a city.</li>

<li>Use as much antenna gain as possible, and get your antenna as high as possible.</li>

<li>Maximum space wave communications distance is the longest distance possible using the choosen antenna heights.&nbsp;&nbsp;It <b>does not</b> mean a link that long is possible.</li>

<li>Try to obtain a thermal noise fade margin of 10 dB, or larger, for a more reliable link.&nbsp;&nbsp;If the fade margin is negative, that link is impossible.</li>
</ul>
</blockquote>

<blockquote>
<p>The radio index of refraction will change with the weather.&nbsp;&nbsp;On more humid days, the value will be greater; on cooler days it will be considerably less.</p>

<p>This calculation assumes K is constant over the whole radio path; that no unusual weather conditions have stratified the air, creating inversions or negative refractive conditions and that the two radio sites are at about the same elevation.&nbsp;&nbsp;These assumptions hold true only for short paths under average or &quot;normal&quot; atmospheric conditions and may not apply at all times.&nbsp;&nbsp;A more accurate determination of the refractive properties over a radio path require data from several points along the radio path.</p>

<p>From SoftWright:</p>

<p><em>Even Fresnel Zone Reflection Points</em></p>

<p>Another important use of Fresnel zone information is to check paths (particularly microwave paths) for possible reflection points.&nbsp;&nbsp;The Fresnel zone formula shown above is the set of points where the distance from the transmitter to the Fresnel zone, then to the receiver, is longer than the direct path from the transmitter to the receiver.&nbsp;&nbsp;For even numbered Fresnel zones (N=2, 4, etc.), the difference between the direct path and the indirect path defined by the Fresnel zone distance, is a multiple of one-half wavelength.&nbsp;&nbsp;If the geometry of the path is such that an even numbered Fresnel zone happens to be tangential to a good reflecting surface (a lake, a highway, a smooth desert area, depending on what wavelength is involved), signal cancellation will occur due to interference between the direct and indirect (reflected) signal paths.</p>

<p>You can set the Fresnel zone to even numbered values when plotting a profile to see if any potential areas of destructive signal reflection are present on the path.</p>
</blockquote>
EOF

$step = ($dist_km * 1000) / 50; # meters

print "<blockquote><p><b><u>Fresnel Zone Plots</u></b></p>";
printf "Step size used : %.2f meters (%.2f feet)\n", $step, $step / 0.3048;

print "<pre><b>Path distance<br>Transmitter to Receiver         0.6 Fresnel Zone Boundary       $nth$sub Fresnel Zone Boundary</b><br><hr noshade>";

print "0.00 km (0.00 mi)\t\t0.00 m (0.00 ft)\t\t0.00 m (0.00 ft)<br>";
for ($pl = $step; $pl < ($dist_km * 1000); $pl = $pl + $step) {
  $fn = 17.31 * sqrt ((0.6 * ($pl * (($dist_km * 1000) - $pl))) / ($frq_mhz * ($dist_km * 1000)));
  $fnn = 17.31 * sqrt (($nth * ($pl * (($dist_km * 1000) - $pl))) / ($frq_mhz * ($dist_km * 1000)));
  printf "%.2f km (%.2f mi)\t\t%.2f m (%.2f ft)\t\t%.2f m (%.2f ft)<br>", $pl / 1000, $pl / 1609.344, $fn, $fn / 0.3048, $fnn, $fnn / 0.3048;
}
print "$dist_km km ($dist_mi mi)\t\t0.00 m (0.00 ft)\t\t0.00 m (0.00 ft)<br>";


print "<hr noshade></pre></blockquote><font size=\"-1\">Calculated on $date GMT</font></body></html>";

